/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2021-2022 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::patchToPatches::inverseDistance

Description
    Class to generate patchToPatch coupling geometry. Couples a face to the
    opposite face onto which its centre-normal ray projects, plus the immediate
    neighbours to that opposite face. The proportion of contribution from the
    different faces is calculated using inverse distance weighting.

SourceFiles
    inverseDistance.C

\*---------------------------------------------------------------------------*/

#ifndef inverseDistancePatchToPatch_H
#define inverseDistancePatchToPatch_H

#include "nearbyPatchToPatch.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace patchToPatches
{

/*---------------------------------------------------------------------------*\
                          Class inverseDistance Declaration
\*---------------------------------------------------------------------------*/

class inverseDistance
:
    public nearby
{
    // Private Member Data

        //- For each source face, the coupled target weights
        List<DynamicList<scalar>> srcWeights_;

        //- For each target face, the coupled source weights
        List<DynamicList<scalar>> tgtWeights_;


    // Private Static Member Functions

        //- Return whether a ray starting at point `p` with direction `r` hits
        //  a face `f` with points `ps`
        static bool rayHitsFace
        (
            const point& p,
            const vector& r,
            const face& f,
            const pointField& ps
        );


    // Private Member Functions

        //- Initialisation
        virtual void initialise
        (
            const primitiveOldTimePatch& srcPatch,
            const vectorField& srcPointNormals,
            const vectorField& srcPointNormals0,
            const primitiveOldTimePatch& tgtPatch
        );

        //- Convert the addressing generated by the base class into weights
        //  (and addressing) appropriate for inverse distance interpolation
        void generateWeights
        (
            const primitiveOldTimePatch& srcPatch,
            const primitiveOldTimePatch& tgtPatch
        );

        //- Finalise the intersection locally. Trims the local target patch
        //  so that only parts that actually intersect the source remain.
        //  Returns new-to-old map for trimming target-associated data.
        virtual labelList finaliseLocal
        (
            const primitiveOldTimePatch& srcPatch,
            const vectorField& srcPointNormals,
            const vectorField& srcPointNormals0,
            const primitiveOldTimePatch& tgtPatch
        );

        //- Send data that resulted from an intersection between the source
        //  patch and a distributed source-local-target patch back to the
        //  original target processes
        virtual void rDistributeTgt(const primitiveOldTimePatch& tgtPatch);

        //- Finalise the intersection
        virtual label finalise
        (
            const primitiveOldTimePatch& srcPatch,
            const vectorField& srcPointNormals,
            const vectorField& srcPointNormals0,
            const primitiveOldTimePatch& tgtPatch,
            const transformer& tgtToSrc
        );

        //- For each source face, the coupled target weights
        virtual tmpNrc<List<DynamicList<scalar>>> srcWeights() const;

        //- For each target face, the coupled source weights
        virtual tmpNrc<List<DynamicList<scalar>>> tgtWeights() const;


public:

    //- Runtime type information
    TypeName("inverseDistance");


    // Constructors

        //- Construct from components
        inverseDistance(const bool reverse);


    //- Destructor
    ~inverseDistance();
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace patchToPatches
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
