/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2022-2023 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::processorTopology

Description
    Determines/represents processor-processor connection.
    After instantiation contains the processor-processor connection table
    (globally synchronized on all processors).

Note
    A processorTopology is usually generated by using the \c New
    factory method, which has been wrapped into a separate header to
    minimize dependencies.

Warning
    Does not currently correctly support multiple processor
    patches connecting two processors.

SourceFiles
    processorTopologyNew.H

\*---------------------------------------------------------------------------*/

#ifndef Foam_processorTopology_H
#define Foam_processorTopology_H

#include "labelList.H"
#include "lduSchedule.H"
#include "Map.H"
#include "UPstream.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                      Class processorTopology Declaration
\*---------------------------------------------------------------------------*/

class processorTopology
{
    // Private Data

        //- Map from neighbour proc to patch index
        //- (processor-local information!)
        Map<label> procPatchMap_;

        //- The neighbour processor connections (ascending order) associated
        //- with the local rank.
        //  Possibly demand-driven data.
        mutable labelList procNeighbours_;

        //- The complete processor to processor connection adjacency table.
        //- Globally synchronized information.
        //  Likely demand-driven data.
        mutable labelListList procAdjacencyTable_;

        //- Order in which the patches should be initialised/evaluated
        //- corresponding to the schedule
        lduSchedule patchSchedule_;

        //- The communicator used during creation of the topology
        label comm_;


    // Private Methods Functions

        // Could expose as public...
        // //- Map of neighbour processor to \em local boundary patch index.
        // const Map<label>& procPatchMap() const noexcept
        // {
        //     return procPatchMap_;
        // }

public:

    // Generated Methods

        //- Copy construct
        processorTopology(const processorTopology&) = default;

        //- Move construct
        processorTopology(processorTopology&&) = default;

        //- Copy assignment
        processorTopology& operator=(const processorTopology&) = default;

        //- Move assignment
        processorTopology& operator=(processorTopology&&) = default;


    // Constructors

        //- Default construct (empty)
        processorTopology() noexcept;


    // Static Functions

        //- Factory method to create topology, schedule and proc/patch maps.
        //  Scans the list of patches for processor connections corresponding
        //  to the ProcPatch type.
        template<class ProcPatch, class PatchListType>
        static processorTopology New
        (
            const PatchListType& patches,
            const label comm
        );


    // Member Functions

        //- The communicator used during creation of the topology
        label comm() const noexcept { return comm_; }

        //- The neighbour processor connections (ascending order) associated
        //- with the \em local rank.
        const labelList& procNeighbours() const;

        //- The complete processor to processor connection adjacency table.
        //- Globally synchronized information.
        //  Likely demand-driven data.
        const labelListList& procAdjacency() const;

        //- Which \em local boundary is attached to specified neighbour
        //- processor.
        //  \return -1 if not currently connected to specified processor.
        label procPatchLookup(const label proci) const
        {
            return procPatchMap_.lookup(proci, -1);
        }

        //- Order in which the patches should be initialised/evaluated
        //- corresponding to the schedule
        const lduSchedule& patchSchedule() const noexcept
        {
            return patchSchedule_;
        }
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
