/*
 * This file is part of libbluray
 * Copyright (C) 2010-2017  Petri Hintukainen <phintuka@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package hdmv;

import blues.Log;
import impl.org.bluray.system.RegisterAccessImpl;

public class REG {

    static final int BD_PSR_COUNT = 128;
    static final int BD_GPR_COUNT = 4096;

    static final int PSR_IG_STREAM_ID     = 0;
    static final int PSR_PRIMARY_AUDIO_ID = 1;
    static final int PSR_PG_STREAM        = 2; /* PG TextST and PIP PG TextST stream number */
    static final int PSR_ANGLE_NUMBER     = 3; /* 1..N */
    static final int PSR_TITLE_NUMBER     = 4; /* 1..N  (0 = top menu, 0xffff = first play) */
    static final int PSR_CHAPTER          = 5; /* 1..N  (0xffff = invalid) */
    static final int PSR_PLAYLIST         = 6; /* playlist file name number */
    static final int PSR_PLAYITEM         = 7; /* 0..N-1 (playitem_id) */
    static final int PSR_TIME             = 8; /* presetation time */
    static final int PSR_NAV_TIMER        = 9;
    static final int PSR_SELECTED_BUTTON_ID = 10;
    static final int PSR_MENU_PAGE_ID     = 11;
    static final int PSR_STYLE            = 12;
    static final int PSR_PARENTAL         = 13;
    static final int PSR_SECONDARY_AUDIO_VIDEO = 14;
    static final int PSR_AUDIO_CAP        = 15;
    static final int PSR_AUDIO_LANG       = 16;
    static final int PSR_PG_AND_SUB_LANG  = 17;
    static final int PSR_MENU_LANG        = 18;
    static final int PSR_COUNTRY          = 19;
    static final int PSR_REGION           = 20;
    static final int PSR_OUTPUT_PREFER    = 21;
    static final int PSR_3D_STATUS        = 22;
    static final int PSR_DISPLAY_CAP      = 23;
    static final int PSR_3D_CAP           = 24;
    static final int PSR_UHD_CAP          = 25;
    static final int PSR_UHD_DISPLAY_CAP  = 26;
    static final int PSR_UHD_HDR_PREFER   = 27;
    static final int PSR_UHD_SDR_CONV_PREFER = 28;
    static final int PSR_VIDEO_CAP        = 29;
    static final int PSR_TEXT_CAP         = 30; /* text subtitles */
    static final int PSR_PROFILE_VERSION  = 31; /* player profile and version */
    static final int PSR_BACKUP_PSR4      = 36;
    static final int PSR_BACKUP_PSR5      = 37;
    static final int PSR_BACKUP_PSR6      = 38;
    static final int PSR_BACKUP_PSR7      = 39;
    static final int PSR_BACKUP_PSR8      = 40;
    static final int PSR_BACKUP_PSR10     = 42;
    static final int PSR_BACKUP_PSR11     = 43;
    static final int PSR_BACKUP_PSR12     = 44;

    /*
     * BLURAY_PLAYER_SETTING_AUDIO_CAP (PSR15)
     *
     * Player capability for audio (bitmask)
     */

    /* LPCM */

    /* 48/96kHz (mandatory) */
    static final int BLURAY_ACAP_LPCM_48_96_STEREO_ONLY = 0x0001;  /* LPCM 48kHz and 96kHz stereo */
    static final int BLURAY_ACAP_LPCM_48_96_SURROUND    = 0x0002;  /* LPCM 48kHz and 96kHz surround */

    /* 192kHz (optional) */
    static final int BLURAY_ACAP_LPCM_192_STEREO_ONLY   = 0x0004;  /* LPCM 192kHz stereo */
    static final int BLURAY_ACAP_LPCM_192_SURROUND      = 0x0008;  /* LPCM 192kHz surround */

    /* Dolby Digital Plus */

    /* independent substream (mandatory) */
    static final int BLURAY_ACAP_DDPLUS_STEREO_ONLY     = 0x0010;
    static final int BLURAY_ACAP_DDPLUS_SURROUND        = 0x0020;

    /* dependent substream (optional) */
    static final int BLURAY_ACAP_DDPLUS_DEP_STEREO_ONLY = 0x0040;
    static final int BLURAY_ACAP_DDPLUS_DEP_SURROUND    = 0x0080;

    /* DTS-HD */

    /* Core substream (mandatory) */
    static final int BLURAY_ACAP_DTSHD_CORE_STEREO_ONLY = 0x0100;
    static final int BLURAY_ACAP_DTSHD_CORE_SURROUND    = 0x0200;

    /* Extension substream (optional) */
    static final int BLURAY_ACAP_DTSHD_EXT_STEREO_ONLY  = 0x0400;
    static final int BLURAY_ACAP_DTSHD_EXT_SURROUND     = 0x0800;

    /* Dolby lossless (TrueHD) */

    /* Dolby Digital (mandatory) */
    static final int BLURAY_ACAP_DD_STEREO_ONLY         = 0x1000;
    static final int BLURAY_ACAP_DD_SURROUND            = 0x2000;

    /* MLP (optional) */
    static final int BLURAY_ACAP_MLP_STEREO_ONLY        = 0x4000;
    static final int BLURAY_ACAP_MLP_SURROUND           = 0x8000;

    /*
     * BLURAY_PLAYER_SETTING_REGION_CODE (PSR20)
     *
     * Player region code (integer)
     *
     * Region A: the Americas, East and Southeast Asia, U.S. territories, and Bermuda.
     * Region B: Africa, Europe, Oceania, the Middle East, the Kingdom of the Netherlands,
     *           British overseas territories, French territories, and Greenland.
     * Region C: Central and South Asia, Mongolia, Russia, and the People's Republic of China.
     *
     */
    static final int BLURAY_REGION_A = 1;
    static final int BLURAY_REGION_B = 2;
    static final int BLURAY_REGION_C = 4;

    /*
     * BLURAY_PLAYER_SETTING_OUTPUT_PREFER (PSR21)
     *
     * Output mode preference (integer)
     */
    static final int BLURAY_OUTPUT_PREFER_2D = 0;
    static final int BLURAY_OUTPUT_PREFER_3D = 1;

    /*
     * BLURAY_PLAYER_SETTING_DISPLAY_CAP (PSR23)
     *
     * Display capability (bit mask) and display size
     */
    static final int BLURAY_DCAP_1080p_720p_3D           = 0x01;  /* capable of 1920x1080 23.976Hz and 1280x720 59.94Hz 3D */
    static final int BLURAY_DCAP_720p_50Hz_3D            = 0x02;  /* capable of 1280x720 50Hz 3D */
    static final int BLURAY_DCAP_NO_3D_CLASSES_REQUIRED  = 0x04;  /* 3D glasses are not required */
    static final int BLURAY_DCAP_INTERLACED_3D           = 0x08;  /* */

    /*
     * BLURAY_PLAYER_SETTING_VIDEO_CAP (PSR29)
     *
     * Player capability for video (bit mask)
     */
    static final int BLURAY_VCAP_SECONDARY_HD = 0x01;  /* player can play secondary stream in HD */
    static final int BLURAY_VCAP_25Hz_50Hz    = 0x02;  /* player can play 25Hz and 50Hz video */

    /*
     * BLURAY_PLAYER_SETTING_PLAYER_PROFILE (PSR31)
     *
     * Player profile and version
     *
     * Profile 1, version 1.0: no local storage, no VFS, no internet
     * Profile 1, version 1.1: PiP, VFS, sec. audio, 256MB local storage, no internet
     * Profile 2, version 2.0: BdLive (internet), 1GB local storage
     */
    static final int BLURAY_PLAYER_PROFILE_1_v1_0 = ((0x00 << 16) | (0x0100));   /* Profile 1, version 1.0 (Initial Standard Profile) */
    static final int BLURAY_PLAYER_PROFILE_1_v1_1 = ((0x01 << 16) | (0x0110));   /* Profile 1, version 1.1 (secondary stream support) */
    static final int BLURAY_PLAYER_PROFILE_2_v2_0 = ((0x03 << 16) | (0x0200));   /* Profile 2, version 2.0 (network access, BdLive) */
    static final int BLURAY_PLAYER_PROFILE_3_v2_0 = ((0x08 << 16) | (0x0200));   /* Profile 3, version 2.0 (audio only player) */
    static final int BLURAY_PLAYER_PROFILE_5_v2_4 = ((0x13 << 16) | (0x0240));   /* Profile 5, version 2.4 (3D) */
    static final int BLURAY_PLAYER_PROFILE_6_v3_0 = ((0x00 << 16) | (0x0300));   /* Profile 6, version 3.0 (UHD) */
    static final int BLURAY_PLAYER_PROFILE_6_v3_1 = ((0x00 << 16) | (0x0310));   /* Profile 6, version 3.1 (UHD) */

    static final int BLURAY_PLAYER_PROFILE_3D_FLAG       = 0x100000;
    static final int BLURAY_PLAYER_PROFILE_VERSION_MASK  = 0xffff;


/*
 * Initial values for player status/setting registers (5.8.2).
 *
 * PS in comment indicates player setting -> register can't be changed from movie object code.
 */

static final int[] bd_psr_init = new int[] {
    1,           /*     PSR0:  Interactive graphics stream number */
    0xff,        /*     PSR1:  Primary audio stream number */
    0x0fff0fff,  /*     PSR2:  PG TextST stream number and PiP PG stream number*/
    1,           /*     PSR3:  Angle number */
    0xffff,      /*     PSR4:  Title number */
    0xffff,      /*     PSR5:  Chapter number */
    0,           /*     PSR6:  PlayList ID */
    0,           /*     PSR7:  PlayItem ID */
    0,           /*     PSR8:  Presentation time */
    0,           /*     PSR9:  Navigation timer */
    0xffff,      /*     PSR10: Selected button ID */
    0,           /*     PSR11: Page ID */
    0xff,        /*     PSR12: User style number */
    0xff,        /* PS: PSR13: User age */
    0xffff,      /*     PSR14: Secondary audio stream number and secondary video stream number */
                 /* PS: PSR15: player capability for audio */
    BLURAY_ACAP_LPCM_48_96_SURROUND |
    BLURAY_ACAP_LPCM_192_SURROUND   |
    BLURAY_ACAP_DDPLUS_SURROUND     |
    BLURAY_ACAP_DDPLUS_DEP_SURROUND |
    BLURAY_ACAP_DTSHD_CORE_SURROUND |
    BLURAY_ACAP_DTSHD_EXT_SURROUND  |
    BLURAY_ACAP_DD_SURROUND         |
    BLURAY_ACAP_MLP_SURROUND,

    0xffffff,    /* PS: PSR16: Language code for audio */
    0xffffff,    /* PS: PSR17: Language code for PG and Text subtitles */
    0xffffff,    /* PS: PSR18: Menu description language code */
    0xffff,      /* PS: PSR19: Country code */
                 /* PS: PSR20: Region code */ /* 1 - A, 2 - B, 4 - C */
    BLURAY_REGION_B,
                 /* PS: PSR21: Output mode preference */
    BLURAY_OUTPUT_PREFER_2D,
    0,           /*     PSR22: Stereoscopic status */
    0,           /* PS: PSR23: Display capability */
    0,           /* PS: PSR24: 3D capability */
    0,           /* PS: PSR25: UHD capability */
    0,           /* PS: PSR26: UHD display capability */
    0,           /* PS: PSR27: HDR preference */
    0,           /* PS: PSR28: SDR conversion preference */
                 /* PS: PSR29: player capability for video */
    BLURAY_VCAP_SECONDARY_HD |
    BLURAY_VCAP_25Hz_50Hz,

    0x1ffff,     /* PS: PSR30: player capability for text subtitle */
                 /* PS: PSR31: Player profile and version */
    BLURAY_PLAYER_PROFILE_2_v2_0,
    0,           /*     PSR32 */
    0,           /*     PSR33 */
    0,           /*     PSR34 */
    0,           /*     PSR35 */
    0xffff,      /*     PSR36: backup PSR4 */
    0xffff,      /*     PSR37: backup PSR5 */
    0,           /*     PSR38: backup PSR6 */
    0,           /*     PSR39: backup PSR7 */
    0,           /*     PSR40: backup PSR8 */
    0,           /*     PSR41: */
    0xffff,      /*     PSR42: backup PSR10 */
    0,           /*     PSR43: backup PSR11 */
    0xff,        /*     PSR44: backup PSR12 */
    0,           /*     PSR45: */
    0,           /*     PSR46: */
    0,           /*     PSR47: */
    0xffffffff,  /* PS: PSR48: Characteristic text caps */
    0xffffffff,  /* PS: PSR49: Characteristic text caps */
    0xffffffff,  /* PS: PSR50: Characteristic text caps */
    0xffffffff,  /* PS: PSR51: Characteristic text caps */
    0xffffffff,  /* PS: PSR52: Characteristic text caps */
    0xffffffff,  /* PS: PSR53: Characteristic text caps */
    0xffffffff,  /* PS: PSR54: Characteristic text caps */
    0xffffffff,  /* PS: PSR55: Characteristic text caps */
    0xffffffff,  /* PS: PSR56: Characteristic text caps */
    0xffffffff,  /* PS: PSR57: Characteristic text caps */
    0xffffffff,  /* PS: PSR58: Characteristic text caps */
    0xffffffff,  /* PS: PSR59: Characteristic text caps */
    0xffffffff,  /* PS: PSR60: Characteristic text caps */
    0xffffffff,  /* PS: PSR61: Characteristic text caps */
    /* 62-95:   reserved */
    /* 96-111:  reserved for BD system use */
    /* 112-127: reserved */
};

/*
 * PSR ids for debugging
 */
static final String bd_psr_name[] = new String[] {
    "IG_STREAM_ID",
    "PRIMARY_AUDIO_ID",
    "PG_STREAM",
    "ANGLE_NUMBER",
    "TITLE_NUMBER",
    "CHAPTER",
    "PLAYLIST",
    "PLAYITEM",
    "TIME",
    "NAV_TIMER",
    "SELECTED_BUTTON_ID",
    "MENU_PAGE_ID",
    "STYLE",
    "PARENTAL",
    "SECONDARY_AUDIO_VIDEO",
    "AUDIO_CAP",
    "AUDIO_LANG",
    "PG_AND_SUB_LANG",
    "MENU_LANG",
    "COUNTRY",
    "REGION",
    "OUTPUT_PREFER",
    "3D_STATUS",
    "DISPLAY_CAP",
    "3D_CAP",
    //"PSR_VIDEO_CAP",
};

/*
 * init / free
 */

static public void bd_registers_init()
{
    int[] psr = RegisterAccessImpl.getInstance().rawPSR();
    System.arraycopy(bd_psr_init,0,psr,0,bd_psr_init.length);
}

/*
 * PSR state save / restore
 */

public static void bd_psr_save_state()
{
    int[] psr = RegisterAccessImpl.getInstance().rawPSR();

    /* store registers 4-8 and 10-12 to backup registers */

    System.arraycopy(psr,  4, psr, 36, 5);
    System.arraycopy(psr, 10, psr, 42, 3);
}

public static void bd_psr_reset_backup_registers()
{
    int[] psr = RegisterAccessImpl.getInstance().rawPSR();

    /* init backup registers to default */
    System.arraycopy(bd_psr_init, 36, psr, 36, 5);
    System.arraycopy(bd_psr_init, 42, psr, 42, 3);

}

public static void bd_psr_restore_state()
{
    int[] psr = RegisterAccessImpl.getInstance().rawPSR();


    /* restore backup registers */
    System.arraycopy(psr, 36, psr,  4, 5);
    System.arraycopy(psr, 42, psr, 10, 3);

    /* init backup registers to default */
    System.arraycopy(bd_psr_init, 36, psr, 36, 5);
    System.arraycopy(bd_psr_init, 42, psr, 42, 3);
}

/*
 * GPR read / write
 */

public static int bd_gpr_write(int reg, int val)
{
    if (reg >= BD_GPR_COUNT) {
        BD_DEBUG(DBG_BLURAY, "bd_gpr_write(%d): invalid register\n", reg);
        return -1;
    }

    RegisterAccessImpl.getInstance().setGPR(reg,val);
    return 0;
}

public static int bd_gpr_read(int reg)
{
    if (reg >= BD_GPR_COUNT) {
        BD_DEBUG(DBG_BLURAY, "bd_gpr_read(%d): invalid register\n", reg);
        return 0;
    }

    return RegisterAccessImpl.getInstance().getGPR(reg);
}

/*
 * PSR read / write
 */

public static int bd_psr_read(int reg)
{
    if (reg >= BD_PSR_COUNT) {
        BD_DEBUG(DBG_BLURAY, "bd_psr_read(%d): invalid register\n", reg);
        return -1;
    }
    return RegisterAccessImpl.getInstance().getPSR(reg);
}

public static int bd_psr_setting_write(int reg, int val)
{
    int[] psr = RegisterAccessImpl.getInstance().rawPSR();

    if (reg >= BD_PSR_COUNT) {
        BD_DEBUG(DBG_BLURAY, "bd_psr_write(%d, %d): invalid register\n", reg, val);
        return -1;
    }

    if (psr[reg] == val) {
        BD_DEBUG(DBG_BLURAY, "bd_psr_write(%d, %d): no change in value\n", reg, val);
    } else if ((reg<bd_psr_name.length) && (null!=bd_psr_name[reg])) {
        BD_DEBUG(DBG_BLURAY, "bd_psr_write(): PSR%-4d (%s) 0x%x -> 0x%x\n", reg, bd_psr_name[reg], psr[reg], val);
    } else {
        BD_DEBUG(DBG_BLURAY, "bd_psr_write(): PSR%-4d 0x%x -> 0x%x\n", reg, psr[reg], val);
    }

    RegisterAccessImpl.getInstance().setPSR(reg,val);

    return 0;
}

public static int bd_psr_write(int reg, int val)
{
    if ((reg == 13) ||
        (reg >= 15 && reg <= 21) ||
        (reg >= 23 && reg <= 31) ||
        (reg >= 48 && reg <= 61)) {
      BD_DEBUG(DBG_BLURAY | DBG_CRIT, "bd_psr_write(%d, %d): read-only register !\n", reg, val);
      return -2;
  }

  return bd_psr_setting_write(reg, val);
}

public static int bd_psr_write_bits(int reg, int val, int mask)
{
    int result;

    if (mask == 0xffffffff) {
        return bd_psr_write(reg, val);
    }

    int psr_value = bd_psr_read(reg);
    psr_value = (psr_value & (~mask)) | (val & mask);
    result = bd_psr_write(reg, psr_value);


    return result;
}

/*
 *
 */

public static int psr_init_3D(boolean initial_mode, boolean force)
{
    /* make automatic initialization to fail if app has already changed player profile */
    if (!force) {
        if ((bd_psr_read(PSR_PROFILE_VERSION) & BLURAY_PLAYER_PROFILE_VERSION_MASK) >= 0x0300) {
            BD_DEBUG(DBG_BLURAY | DBG_CRIT, "psr_init_3D() failed: profile version already set to >= 0x0300 (profile 6)\n");
            return -1;
        }
        if ((bd_psr_read(PSR_PROFILE_VERSION) & BLURAY_PLAYER_PROFILE_3D_FLAG) != 0) {
            BD_DEBUG(DBG_BLURAY | DBG_CRIT, "psr_init_3D() failed: 3D already set in profile\n");
            return -1;
        }
    }

    bd_psr_setting_write(PSR_OUTPUT_PREFER,
                         BLURAY_OUTPUT_PREFER_3D);

    bd_psr_setting_write(PSR_DISPLAY_CAP,
                         BLURAY_DCAP_1080p_720p_3D |
                         BLURAY_DCAP_720p_50Hz_3D |
                         BLURAY_DCAP_NO_3D_CLASSES_REQUIRED |
                         BLURAY_DCAP_INTERLACED_3D |
                         0);

    bd_psr_setting_write(PSR_3D_CAP,
                         /* TODO */ 0xffffffff );

    bd_psr_setting_write(PSR_PROFILE_VERSION,
                         BLURAY_PLAYER_PROFILE_5_v2_4);

    bd_psr_write(PSR_3D_STATUS,
                 initial_mode?1:0);

    return 0;
}

public static int psr_init_UHD(boolean force)
{
    /* make automatic initialization to fail if app has already changed player profile */
    if (!force) {
        if ((bd_psr_read(PSR_PROFILE_VERSION) & BLURAY_PLAYER_PROFILE_VERSION_MASK) >= 0x0300) {
          BD_DEBUG(DBG_BLURAY | DBG_CRIT, "psr_init_UHD() failed: profile version already >= 0x0300\n");
          return -1;
        }
        if ((bd_psr_read(PSR_PROFILE_VERSION) & BLURAY_PLAYER_PROFILE_3D_FLAG) != 0) {
            BD_DEBUG(DBG_BLURAY | DBG_CRIT, "psr_init_UHD() failed: 3D already set in profile\n");
            return -1;
        }
    }

    bd_psr_setting_write(PSR_UHD_CAP,
                         /* TODO */ 0xffffffff );

    bd_psr_setting_write(PSR_UHD_DISPLAY_CAP,
                         /* TODO */ 0xffffffff );

    bd_psr_setting_write(PSR_UHD_HDR_PREFER,
                         /* TODO */ 0xffffffff );

    bd_psr_setting_write(PSR_UHD_SDR_CONV_PREFER,
                         /* TODO */ 0 );

    bd_psr_setting_write(PSR_PROFILE_VERSION,
                         BLURAY_PLAYER_PROFILE_6_v3_1);

    return 0;
}

    static final int DBG_BLURAY = 0;
    static final int DBG_CRIT = 1;

    static void BD_DEBUG(int t,String msg) {
        dbgout(msg);
    }

    static void BD_DEBUG(int t,String msg,int p0) {
        dbgout(String.format(msg,p0));
    }

    static void BD_DEBUG(int t,String msg,int p0,int p1) {
        dbgout(String.format(msg,p0,p1));
    }

    static void BD_DEBUG(int t,String msg,int p0,int p1,int p2) {
        dbgout(String.format(msg,p0,p1,p2));
    }

    static void BD_DEBUG(int t,String msg,int p0,String p1,int p2,int p3) {
        dbgout(String.format(msg,p0,p1,p2,p3));
    }

    static void dbgout(String msg) {
        Log.logLF(2,Log.LOG_HDMV, msg);
    }
}