/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.citationstyle;

import de.undercouch.citeproc.helper.CSLUtils;
import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.jabref.logic.util.StandardFileType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.CharacterData;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class CitationStyle {
    public static final String DEFAULT = "/ieee.csl";
    private static final Logger LOGGER = LoggerFactory.getLogger(CitationStyle.class);
    private static final String STYLES_ROOT = "/csl-styles";
    private static final List<CitationStyle> STYLES = new ArrayList<CitationStyle>();
    private static final DocumentBuilderFactory FACTORY = DocumentBuilderFactory.newInstance();
    private final String filePath;
    private final String title;
    private final String source;

    private CitationStyle(String filename, String title, String source) {
        this.filePath = Objects.requireNonNull(filename);
        this.title = Objects.requireNonNull(title);
        this.source = Objects.requireNonNull(source);
    }

    private static Optional<CitationStyle> createCitationStyleFromSource(String source, String filename) {
        if (filename != null && !filename.isEmpty() && source != null && !source.isEmpty()) {
            try {
                InputSource inputSource = new InputSource();
                inputSource.setCharacterStream(new StringReader(CitationStyle.stripInvalidProlog(source)));
                Document doc = FACTORY.newDocumentBuilder().parse(inputSource);
                NodeList bibs = doc.getElementsByTagName("bibliography");
                if (bibs.getLength() <= 0) {
                    LOGGER.debug("no bibliography element for file {} ", (Object)filename);
                    return Optional.empty();
                }
                NodeList nodes = doc.getElementsByTagName("info");
                NodeList titleNode = ((Element)nodes.item(0)).getElementsByTagName("title");
                String title = ((CharacterData)titleNode.item(0).getFirstChild()).getData();
                return Optional.of(new CitationStyle(filename, title, source));
            }
            catch (IOException | ParserConfigurationException | SAXException e) {
                LOGGER.error("Error while parsing source", (Throwable)e);
            }
        }
        return Optional.empty();
    }

    private static String stripInvalidProlog(String source) {
        int startIndex = source.indexOf("<");
        if (startIndex > 0) {
            return source.substring(startIndex);
        }
        return source;
    }

    public static Optional<CitationStyle> createCitationStyleFromFile(String styleFile) {
        if (!CitationStyle.isCitationStyleFile(styleFile)) {
            LOGGER.error("Can only load style files: {}", (Object)styleFile);
            return Optional.empty();
        }
        try {
            String internalFile = STYLES_ROOT + (styleFile.startsWith("/") ? "" : "/") + styleFile;
            URL url = CitationStyle.class.getResource(internalFile);
            String text = url != null ? CSLUtils.readURLToString((URL)url, (String)StandardCharsets.UTF_8.toString()) : Files.readString(Path.of(styleFile, new String[0]));
            return CitationStyle.createCitationStyleFromSource(text, styleFile);
        }
        catch (NoSuchFileException e) {
            LOGGER.error("Could not find file: {}", (Object)styleFile, (Object)e);
        }
        catch (IOException e) {
            LOGGER.error("Error reading source file", (Throwable)e);
        }
        return Optional.empty();
    }

    public static CitationStyle getDefault() {
        return CitationStyle.createCitationStyleFromFile(DEFAULT).orElse(new CitationStyle("", "Empty", ""));
    }

    public static List<CitationStyle> discoverCitationStyles() {
        if (!STYLES.isEmpty()) {
            return STYLES;
        }
        URL url = CitationStyle.class.getResource("/csl-styles/ieee.csl");
        if (url == null) {
            LOGGER.error("Could not find any citation style. Tried with {}.", (Object)DEFAULT);
            return Collections.emptyList();
        }
        try {
            URI uri = url.toURI();
            Path path = Path.of(uri).getParent();
            STYLES.addAll(CitationStyle.discoverCitationStylesInPath(path));
            return STYLES;
        }
        catch (IOException | URISyntaxException e) {
            LOGGER.error("something went wrong while searching available CitationStyles", (Throwable)e);
            return Collections.emptyList();
        }
    }

    private static List<CitationStyle> discoverCitationStylesInPath(Path path) throws IOException {
        try (Stream<Path> stream = Files.find(path, 1, (file, attr) -> file.toString().endsWith("csl"), new FileVisitOption[0]);){
            List<CitationStyle> list = stream.map(Path::getFileName).map(Path::toString).map(CitationStyle::createCitationStyleFromFile).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
            return list;
        }
    }

    public static boolean isCitationStyleFile(String styleFile) {
        return StandardFileType.CITATION_STYLE.getExtensions().stream().anyMatch(styleFile::endsWith);
    }

    public String getTitle() {
        return this.title;
    }

    public String getSource() {
        return this.source;
    }

    public String getFilePath() {
        return this.filePath;
    }

    public String toString() {
        return this.title;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        CitationStyle other = (CitationStyle)o;
        return Objects.equals(this.source, other.source);
    }

    public int hashCode() {
        return Objects.hash(this.source);
    }
}

