// -*- mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// vi: set et ts=4 sw=4 sts=4:
//
// SPDX-FileCopyrightText: Copyright © DuMux Project contributors, see AUTHORS.md in root folder
// SPDX-License-Identifier: GPL-3.0-or-later
//
/*!
 * \file
 * \ingroup OnePNCModel
 * \brief  Adaption of the fully implicit model to the one-phase n-component flow model.
 *
 * This model implements a one-phase flow of a compressible fluid, that consists
 * of n components, using a standard Darcy approach as the equation for the
 * conservation of momentum. For details on Darcy's law see dumux/flux/darcyslaw.hh.
 * Gravity can be enabled or disabled via the property system.
 * By inserting Darcy's law into the continuity equation, one gets
 \f[
 \frac{\partial (\phi \varrho) }{\partial t} - \nabla \cdot \left\{
   \varrho \frac{\textbf K}{\mu}  \left(\nabla  p - \varrho {\textbf g} \right)
 \right\} = q.
 \f]
 *
 * The transport of the components \f$\kappa \in \{ w, a, ... \}\f$ is described by the following equation:
 \f[
 \frac{ \partial (\phi \varrho X^\kappa) }{\partial t}
 - \nabla \cdot \left\lbrace \varrho X^\kappa \frac{{\textbf K}}{\mu} \left( \nabla  p -
 \varrho {\textbf g} \right)
 + \varrho D^\kappa_\text{pm} \nabla X^\kappa \right\rbrace = q,
 \f]
 *
  * where:
 * * \f$ \phi \f$ is the porosity of the porous medium,
 * * \f$ \varrho \f$ is the mass density,
 * * \f$ X^\kappa \f$ is the mass fraction of component \f$ \kappa \f$,
 * * \f$ \textbf{K} \f$ is the intrinsic permeability tensor,
 * * \f$ \mu \f$ represents the dynamic viscosity,
 * * \f$ p \f$ is the pressure,
 * * \f$ \textbf{g} \f$ is the gravitational acceleration vector,
 * * \f$ {\bf D_{pm}^\kappa} \f$ is the effective diffusivity in the porous medium,
 * * and \f$ q \f$ is a source or sink term.
 *
 * The model is able to use either mole or mass fractions. The property useMoles can be set to either true or false in the
 * problem file. Make sure that the according units are used in the problem setup. useMoles is set to true by default.
 *
 * The primary variables are the pressure \f$p\f$ and the mole fraction of dissolved components \f$x^\kappa\f$.
 */

#ifndef DUMUX_1PNC_MODEL_HH
#define DUMUX_1PNC_MODEL_HH

#include <dumux/common/properties.hh>

#include <dumux/material/fluidmatrixinteractions/thermalconductivityaverage.hh>
#include <dumux/material/fluidstates/compositional.hh>

#include <dumux/porousmediumflow/properties.hh>
#include <dumux/porousmediumflow/1p/model.hh>
#include <dumux/porousmediumflow/compositional/localresidual.hh>
#include <dumux/porousmediumflow/nonisothermal/model.hh>
#include <dumux/porousmediumflow/nonisothermal/indices.hh>
#include <dumux/porousmediumflow/nonisothermal/iofields.hh>
#include <dumux/porousmediumflow/nonequilibrium/model.hh>
#include <dumux/porousmediumflow/nonequilibrium/volumevariables.hh>
#include <dumux/material/fluidmatrixinteractions/diffusivitymillingtonquirk.hh>

#include "indices.hh"
#include "volumevariables.hh"
#include "iofields.hh"

namespace Dumux {

/*!
 * \ingroup OnePNCModel
 * \brief Specifies a number properties of models that
 *        consider a single-phase with multiple components.
 *
 * \tparam nComp the number of components to be considered.
 */
template<int nComp, bool useM, int enableCompDisp, int enableThermDisp, int repCompEqIdx = nComp, class CDM = void>
struct OnePNCModelTraits
{
    using Indices = OnePNCIndices;
    using CompositionalDispersionModel = CDM;

    static constexpr int numEq() { return nComp; }
    static constexpr int numFluidPhases() { return 1; }
    static constexpr int numFluidComponents() { return nComp; }
    static constexpr int replaceCompEqIdx() { return repCompEqIdx; }

    static constexpr bool useMoles() { return useM; }
    static constexpr bool enableAdvection() { return true; }
    static constexpr bool enableMolecularDiffusion() { return true; }
    static constexpr bool enableCompositionalDispersion() { return enableCompDisp; }
    static constexpr bool enableThermalDispersion() { return enableThermDisp; }
    static constexpr bool enableEnergyBalance() { return false; }
};

namespace Properties {

//////////////////////////////////////////////////////////////////
// Type tags
//////////////////////////////////////////////////////////////////

//! The type tag for the implicit the isothermal & non-isothermal one phase n component problems
// Create new type tags
namespace TTag {
struct OnePNC { using InheritsFrom = std::tuple<PorousMediumFlow>; };
struct OnePNCNI { using InheritsFrom = std::tuple<OnePNC>; };
} // end namespace TTag

///////////////////////////////////////////////////////////////////////////
// Properties for the isothermal single phase model
///////////////////////////////////////////////////////////////////////////

//! Set as default that no component mass balance is replaced by the total mass balance
template<class TypeTag>
struct ReplaceCompEqIdx<TypeTag, TTag::OnePNC> { static constexpr int value = GetPropType<TypeTag, Properties::FluidSystem>::numComponents; };

//! The base model traits. Per default, we use the number of components of the fluid system.
template<class TypeTag>
struct BaseModelTraits<TypeTag, TTag::OnePNC>
{
private:
    using FluidSystem = GetPropType<TypeTag, Properties::FluidSystem>;
    using CDM = GetPropType<TypeTag, Properties::CompositionalDispersionModel>;
public:
    using type = OnePNCModelTraits<FluidSystem::numComponents,
                                   getPropValue<TypeTag, Properties::UseMoles>(),
                                   getPropValue<TypeTag, Properties::EnableCompositionalDispersion>(),
                                   getPropValue<TypeTag, Properties::EnableThermalDispersion>(),
                                   getPropValue<TypeTag, Properties::ReplaceCompEqIdx>(),
                                   CDM>;
};
template<class TypeTag>
struct ModelTraits<TypeTag, TTag::OnePNC> { using type = GetPropType<TypeTag, Properties::BaseModelTraits>; }; //!< default the actually used traits to the base traits

/*!
 * \brief The fluid state which is used by the volume variables to
 *        store the thermodynamic state.
 *
 * This should be chosen appropriately for the model ((non-)isothermal,
 * equilibrium, ...). This can be done in the problem.
 */
template<class TypeTag>
struct FluidState<TypeTag, TTag::OnePNC>
{
private:
    using Scalar = GetPropType<TypeTag, Properties::Scalar>;
    using FluidSystem = GetPropType<TypeTag, Properties::FluidSystem>;
    using ModelTraits = GetPropType<TypeTag, Properties::ModelTraits>;
public:
    using type = CompositionalFluidState<Scalar, FluidSystem>;
};

//! Use the model after Millington (1961) for the effective diffusivity
template<class TypeTag>
struct EffectiveDiffusivityModel<TypeTag, TTag::OnePNC>
{ using type = DiffusivityMillingtonQuirk<GetPropType<TypeTag, Properties::Scalar>>; };

//! Use mole fractions in the balance equations by default
template<class TypeTag>
struct UseMoles<TypeTag, TTag::OnePNC> { static constexpr bool value = true; };

//! The local residual function
template<class TypeTag>
struct LocalResidual<TypeTag, TTag::OnePNC> { using type = CompositionalLocalResidual<TypeTag>; };

//! Set the volume variables property
template<class TypeTag>
struct VolumeVariables<TypeTag, TTag::OnePNC>
{
private:
    using PV = GetPropType<TypeTag, Properties::PrimaryVariables>;
    using FSY = GetPropType<TypeTag, Properties::FluidSystem>;
    using FST = GetPropType<TypeTag, Properties::FluidState>;
    using SSY = GetPropType<TypeTag, Properties::SolidSystem>;
    using SST = GetPropType<TypeTag, Properties::SolidState>;
    using PT = typename GetPropType<TypeTag, Properties::SpatialParams>::PermeabilityType;
    using MT = GetPropType<TypeTag, Properties::ModelTraits>;
    static_assert(FSY::numComponents == MT::numFluidComponents(), "Number of components mismatch between model and fluid system");
    static_assert(FST::numComponents == MT::numFluidComponents(), "Number of components mismatch between model and fluid state");
    static_assert(FSY::numPhases == MT::numFluidPhases(), "Number of phases mismatch between model and fluid system");
    static_assert(FST::numPhases == MT::numFluidPhases(), "Number of phases mismatch between model and fluid state");
    using BaseTraits = OnePVolumeVariablesTraits<PV, FSY, FST, SSY, SST, PT, MT>;

    using DT = GetPropType<TypeTag, Properties::MolecularDiffusionType>;
    using EDM = GetPropType<TypeTag, Properties::EffectiveDiffusivityModel>;
    template<class BaseTraits, class DT, class EDM>
    struct NCTraits : public BaseTraits
    {
        using DiffusionType = DT;
        using EffectiveDiffusivityModel = EDM;
    };

public:
    using type = OnePNCVolumeVariables<NCTraits<BaseTraits, DT, EDM>>;
};

//! Set the vtk output fields specific to this model
template<class TypeTag>
struct IOFields<TypeTag, TTag::OnePNC> { using type = OnePNCIOFields; };

///////////////////////////////////////////////////////////////////////////
// Properties for the non-isothermal single phase model
///////////////////////////////////////////////////////////////////////////

//! The non-isothermal vtk output fields
template<class TypeTag>
struct IOFields<TypeTag, TTag::OnePNCNI> { using type = EnergyIOFields<OnePNCIOFields>; };

//! Use the average for effective conductivities
template<class TypeTag>
struct ThermalConductivityModel<TypeTag, TTag::OnePNCNI>
{ using type = ThermalConductivityAverage<GetPropType<TypeTag, Properties::Scalar>>; };

//! Model traits of the non-isothermal model.
template<class TypeTag>
struct ModelTraits<TypeTag, TTag::OnePNCNI>
{
private:
    using IsothermalTraits = GetPropType<TypeTag, Properties::BaseModelTraits>;
    using TDM = GetPropType<TypeTag, Properties::ThermalDispersionModel>;
public:
    using type = PorousMediumFlowNIModelTraits<IsothermalTraits, TDM>;
};

template<class TypeTag>
struct VolumeVariables<TypeTag, TTag::OnePNCNI>
{
private:
    using PV = GetPropType<TypeTag, Properties::PrimaryVariables>;
    using FSY = GetPropType<TypeTag, Properties::FluidSystem>;
    using FST = GetPropType<TypeTag, Properties::FluidState>;
    using SSY = GetPropType<TypeTag, Properties::SolidSystem>;
    using SST = GetPropType<TypeTag, Properties::SolidState>;
    using MT = GetPropType<TypeTag, Properties::ModelTraits>;
    using PT = typename GetPropType<TypeTag, Properties::SpatialParams>::PermeabilityType;
    static_assert(FSY::numComponents == MT::numFluidComponents(), "Number of components mismatch between model and fluid system");
    static_assert(FST::numComponents == MT::numFluidComponents(), "Number of components mismatch between model and fluid state");
    static_assert(FSY::numPhases == MT::numFluidPhases(), "Number of phases mismatch between model and fluid system");
    static_assert(FST::numPhases == MT::numFluidPhases(), "Number of phases mismatch between model and fluid state");
    using BaseTraits = OnePVolumeVariablesTraits<PV, FSY, FST, SSY, SST, PT, MT>;

    using DT = GetPropType<TypeTag, Properties::MolecularDiffusionType>;
    using EDM = GetPropType<TypeTag, Properties::EffectiveDiffusivityModel>;
    using ETCM = GetPropType< TypeTag, Properties:: ThermalConductivityModel>;
    template<class BaseTraits, class DT, class EDM, class ETCM>
    struct NCNITraits : public BaseTraits
    {
        using DiffusionType = DT;
        using EffectiveDiffusivityModel = EDM;
        using EffectiveThermalConductivityModel = ETCM;
    };

public:
    using type = OnePNCVolumeVariables<NCNITraits<BaseTraits, DT, EDM, ETCM>>;
};

} // end namespace Properties

template<class OnePNCModelTraits>
struct OnePNCUnconstrainedModelTraits : public OnePNCModelTraits
{
    static constexpr int numConstraintEq() { return 0; }
};

namespace Properties {
//////////////////////////////////////////////////////////////////
// Type tags
//////////////////////////////////////////////////////////////////
namespace TTag {
struct OnePNCNonEquil { using InheritsFrom = std::tuple<NonEquilibrium, OnePNC>; };
} // end namespace TTag


/////////////////////////////////////////////////
// Properties for the non-equilibrium OnePNC model
/////////////////////////////////////////////////

template<class TypeTag>
struct EquilibriumLocalResidual<TypeTag, TTag::OnePNCNonEquil> { using type = CompositionalLocalResidual<TypeTag>; };

//! Set the vtk output fields specific to this model
template<class TypeTag>
struct EquilibriumIOFields<TypeTag, TTag::OnePNCNonEquil> { using type = OnePNCIOFields; };

template<class TypeTag>
struct ModelTraits<TypeTag, TTag::OnePNCNonEquil>
{
private:
    using EquiTraits = GetPropType<TypeTag, Properties::EquilibriumModelTraits>;
    static constexpr bool enableTNE = getPropValue<TypeTag, Properties::EnableThermalNonEquilibrium>();
    static constexpr bool enableCNE = getPropValue<TypeTag, Properties::EnableChemicalNonEquilibrium>();
    static constexpr int numEF = getPropValue<TypeTag, Properties::NumEnergyEqFluid>();
    static constexpr int numES = getPropValue<TypeTag, Properties::NumEnergyEqSolid>();
    static constexpr auto nf = getPropValue<TypeTag, Properties::NusseltFormulation>();
    static constexpr auto ns = getPropValue<TypeTag, Properties::SherwoodFormulation>();

    using NonEquilTraits = NonEquilibriumModelTraits<EquiTraits, enableCNE, enableTNE, numEF, numES, nf, ns>;
public:
    using type = NonEquilTraits;
};

// by default chemical non equilibrium is enabled in the nonequil model, switch that off here
template<class TypeTag>
struct EnableChemicalNonEquilibrium<TypeTag, TTag::OnePNCNonEquil> { static constexpr bool value = false; };

//! Set equilibrium model traits
template<class TypeTag>
struct EquilibriumModelTraits<TypeTag, TTag::OnePNCNonEquil>
{
private:
     using FluidSystem = GetPropType<TypeTag, Properties::FluidSystem>;
     using EquilibriumTraits = OnePNCModelTraits<FluidSystem::numComponents,
                                                 getPropValue<TypeTag, Properties::UseMoles>(),
                                                 getPropValue<TypeTag, Properties::EnableCompositionalDispersion>(),
                                                 getPropValue<TypeTag, Properties::EnableThermalDispersion>(),
                                                 getPropValue<TypeTag, Properties::ReplaceCompEqIdx>()>;
public:
    using type = OnePNCUnconstrainedModelTraits<EquilibriumTraits>;
};

//! In case we do not assume full non-equilibrium one needs a thermal conductivity
template<class TypeTag>
struct ThermalConductivityModel<TypeTag, TTag::OnePNCNonEquil>
{
private:
    using Scalar = GetPropType<TypeTag, Properties::Scalar>;
public:
    using type = ThermalConductivityAverage<Scalar>;
};

//! Use the mineralization volume variables together with the 2pnc vol vars
template<class TypeTag>
struct VolumeVariables<TypeTag, TTag::OnePNCNonEquil>
{
private:
    using PV = GetPropType<TypeTag, Properties::PrimaryVariables>;
    using FSY = GetPropType<TypeTag, Properties::FluidSystem>;
    using FST = GetPropType<TypeTag, Properties::FluidState>;
    using SSY = GetPropType<TypeTag, Properties::SolidSystem>;
    using SST = GetPropType<TypeTag, Properties::SolidState>;
    using MT = GetPropType<TypeTag, Properties::ModelTraits>;
    using PT = typename GetPropType<TypeTag, Properties::SpatialParams>::PermeabilityType;
    using BaseTraits = OnePVolumeVariablesTraits<PV, FSY, FST, SSY, SST, PT, MT>;

    using DT = GetPropType<TypeTag, Properties::MolecularDiffusionType>;
    using EDM = GetPropType<TypeTag, Properties::EffectiveDiffusivityModel>;
    using ETCM = GetPropType< TypeTag, Properties:: ThermalConductivityModel>;
    template<class BaseTraits, class DT, class EDM, class ETCM>
    struct NCNITraits : public BaseTraits
    {
        using DiffusionType = DT;
        using EffectiveDiffusivityModel = EDM;
        using EffectiveThermalConductivityModel = ETCM;
    };

    using EquilibriumVolVars = OnePNCVolumeVariables<NCNITraits<BaseTraits, DT, EDM, ETCM>>;
public:
    using type = NonEquilibriumVolumeVariables<NCNITraits<BaseTraits, DT, EDM, ETCM>, EquilibriumVolVars>;
};

} // end namespace Properties
} // end namespace Dumux

#endif
