# This code is part of Qiskit.
#
# (C) Copyright IBM 2021.
#
# This code is licensed under the Apache License, Version 2.0. You may
# obtain a copy of this license in the LICENSE.txt file in the root directory
# of this source tree or at http://www.apache.org/licenses/LICENSE-2.0.
#
# Any modifications or derivative works of this code must retain this
# copyright notice, and modified files need to carry a notice indicating
# that they have been altered from the originals.

"""Backend run options."""

from dataclasses import asdict, dataclass
from typing import Any
from packaging.version import Version

from qiskit import QuantumCircuit, __version__ as qiskit_version

if Version(qiskit_version).major >= 2:
    from qiskit.result import MeasLevel, MeasReturnType
else:
    from qiskit.qobj.utils import MeasLevel, MeasReturnType  # pylint: disable=import-error


@dataclass
class CommonOptions:
    """Options common for both paths."""

    shots: int = 4000
    meas_level: int | MeasLevel = MeasLevel.CLASSIFIED
    init_qubits: bool = True
    rep_delay: float | None = None
    memory: bool = False
    meas_return: str | MeasReturnType = MeasReturnType.AVERAGE

    def to_transport_dict(self) -> dict[str, Any]:
        """Remove None values so runtime defaults are used."""
        dict_ = asdict(self)
        for key in list(dict_.keys()):
            if dict_[key] is None:
                del dict_[key]
        return dict_


@dataclass
class QASM3Options(CommonOptions):
    """Options for the QASM3 path."""

    init_circuit: QuantumCircuit | None = None
    init_num_resets: int | None = None


@dataclass
class QASM2Options(CommonOptions):
    """Options for the QASM2 path."""

    header: dict | None = None
    init_qubits: bool = True
    use_measure_esp: bool | None = None
    # Simulator only
    noise_model: Any = None
    seed_simulator: int | None = None
