/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.time.planet;

import gov.nasa.giss.math.Point3DCartesian;
import gov.nasa.giss.math.PointLL;
import gov.nasa.giss.time.ClockType;
import gov.nasa.giss.time.HourFormatter;
import gov.nasa.giss.time.JulianDate;
import gov.nasa.giss.time.TimeEvent;
import gov.nasa.giss.time.TimeListener;
import gov.nasa.giss.time.TimeUnits;
import gov.nasa.giss.time.planet.DistanceUnits;
import gov.nasa.giss.time.planet.EarthTime;
import java.awt.geom.Point2D;
import java.lang.invoke.MethodHandles;
import java.text.DecimalFormat;
import javax.swing.event.EventListenerList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractPlanetTime
extends JulianDate {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    protected static final DecimalFormat FORMAT_05 = new DecimalFormat("0.00000");
    private static final DecimalFormat FORMAT_06 = new DecimalFormat("0.000000");
    protected static final DecimalFormat FORMAT_20 = new DecimalFormat("00");
    protected double flattening_;
    protected double oneMinusFlatten_;
    protected double semimajorAxisAU_;
    protected double meanAnomaly_;
    protected double meanAnomalyRad_;
    protected double trueAnomaly_;
    protected double obliquity_;
    protected double eccentricity_;
    protected double veeM_;
    protected double ellS_;
    protected double ellSRad_;
    protected double cosEllS_;
    protected double sinEllS_;
    protected double ellSP_;
    protected double alphaFMS_;
    protected double subsolarLonPG_;
    protected double declinSPG_;
    protected double rHeliocentricAU_;
    protected double lonEcliptic_;
    protected double latEcliptic_;
    protected double eotDegrees_;
    protected double eotHours_;
    protected double mst0_;
    protected double tst0_;
    protected String pmTimeSuffix_ = "XT";
    protected double lambdaRot_;
    protected double betaRot_;
    protected EarthTime earthTime_;
    protected double distEM_;
    protected double subEarthLonPG_;
    protected double declinEPG_;
    protected boolean gotOrbitParams_;
    protected boolean gotEarthParams_;
    private final EventListenerList listenerList_ = new EventListenerList();

    public AbstractPlanetTime() {
        this(System.currentTimeMillis());
    }

    public AbstractPlanetTime(long millis) {
        super(millis);
    }

    @Override
    public void setMillis(long millis) {
        super.setMillis(millis);
        this.calculate();
        this.fireTimeChanged();
    }

    protected abstract void calculate();

    protected abstract void calculateOrbitParams();

    protected void calculateEarthParams() {
        if (!this.gotOrbitParams_) {
            this.calculateOrbitParams();
        }
        double latEclipticRad = this.latEcliptic_ * (Math.PI / 180);
        double cosLatEcliptic = Math.cos(latEclipticRad);
        double sinLatEcliptic = Math.sin(latEclipticRad);
        this.earthTime_ = new EarthTime(this.getMillis());
        double lambdaE = this.earthTime_.getHeliocentricLongitude();
        double rE = this.earthTime_.getDistanceToSun(DistanceUnits.AU);
        double lambda = AbstractPlanetTime.normalize180(this.lonEcliptic_ - lambdaE);
        double cosLambda = Math.cos(lambda * (Math.PI / 180));
        double rMu = this.rHeliocentricAU_ * cosLatEcliptic;
        double zEcliptic = this.rHeliocentricAU_ * sinLatEcliptic;
        double rE2 = rE * rE;
        double rM2 = this.rHeliocentricAU_ * this.rHeliocentricAU_;
        double rMu2 = rMu * rMu;
        double distEM2 = rE2 + rM2 - 2.0 * rE * this.rHeliocentricAU_ * cosLambda * cosLatEcliptic;
        this.distEM_ = Math.sqrt(distEM2);
        double distEMu2 = rE2 + rM2 * cosLatEcliptic * cosLatEcliptic - 2.0 * rE * this.rHeliocentricAU_ * cosLambda * cosLatEcliptic;
        double distEMu = Math.sqrt(distEMu2);
        double cosPhase = (rMu2 + distEMu2 - rE2) / (2.0 * rMu * distEMu);
        double phaseRad = Math.acos(cosPhase);
        if (lambda < 0.0) {
            phaseRad = -phaseRad;
        }
        double sinPhase = Math.sin(phaseRad);
        double cosChi = zEcliptic / this.distEM_;
        double chiNRad = Math.PI - Math.acos(cosChi);
        double sinChiN = Math.sin(chiNRad);
        Point3DCartesian pointE = new Point3DCartesian(sinChiN * cosPhase, sinChiN * sinPhase, Math.cos(chiNRad));
        Point3DCartesian pointS = new Point3DCartesian(cosLatEcliptic, 0.0, -sinLatEcliptic);
        pointE.rotateAxis(Point3DCartesian.Axis.Z, this.lambdaRot_ - this.lonEcliptic_, Point3DCartesian.AngularUnits.DEGREES);
        pointS.rotateAxis(Point3DCartesian.Axis.Z, this.lambdaRot_ - this.lonEcliptic_, Point3DCartesian.AngularUnits.DEGREES);
        pointE.rotateAxis(Point3DCartesian.Axis.Y, -(90.0 - this.betaRot_), Point3DCartesian.AngularUnits.DEGREES);
        pointS.rotateAxis(Point3DCartesian.Axis.Y, -(90.0 - this.betaRot_), Point3DCartesian.AngularUnits.DEGREES);
        double declinEPC_ = 90.0 - pointE.getPhi();
        this.declinEPG_ = this.transformLat2PG(declinEPC_);
        this.subEarthLonPG_ = AbstractPlanetTime.normalize360(pointS.getTheta() - pointE.getTheta() + this.subsolarLonPG_);
        this.gotEarthParams_ = true;
    }

    public abstract double getVernalEquinox();

    public double getEccentricity() {
        return this.eccentricity_;
    }

    public double getMeanAnomaly() {
        return this.meanAnomaly_;
    }

    public double getTrueAnomaly() {
        return this.trueAnomaly_;
    }

    public double getHeliocentricLongitude() {
        if (!this.gotOrbitParams_) {
            this.calculateOrbitParams();
        }
        return this.lonEcliptic_;
    }

    public abstract double getHeliocentricLongitudeAtPerihelion();

    public double getHeliocentricLatitude() {
        if (!this.gotOrbitParams_) {
            this.calculateOrbitParams();
        }
        return this.latEcliptic_;
    }

    public double getDistanceToSun(DistanceUnits units) {
        if (!this.gotOrbitParams_) {
            this.calculateOrbitParams();
        }
        switch (units) {
            case AU: {
                return this.rHeliocentricAU_;
            }
            case KM: {
                return this.rHeliocentricAU_ * 1.495978707E8;
            }
            case LIGHTSEC: {
                return this.rHeliocentricAU_ * 499.0047838361564;
            }
            case LIGHTMIN: {
                return this.rHeliocentricAU_ * 499.0047838361564 / 60.0;
            }
        }
        throw new IllegalArgumentException("No switch case for DistanceUnits");
    }

    public double getDistanceToSun(DistanceUnits units, double d) {
        double e = this.getEccentricity();
        double au = this.semimajorAxisAU_ * (1.0 - e * e) / (1.0 + e * Math.cos(d * (Math.PI / 180)));
        switch (units) {
            case AU: {
                return au;
            }
            case KM: {
                return au * 1.495978707E8;
            }
            case LIGHTSEC: {
                return au * 499.0047838361564;
            }
            case LIGHTMIN: {
                return au * 499.0047838361564 / 60.0;
            }
        }
        throw new IllegalArgumentException("No switch case for DistanceUnits");
    }

    public double getEarthDistance(DistanceUnits units) {
        if (!this.gotEarthParams_) {
            this.calculateEarthParams();
        }
        switch (units) {
            case AU: {
                return this.distEM_;
            }
            case KM: {
                return this.distEM_ * 1.495978707E8;
            }
            case LIGHTSEC: {
                return this.distEM_ * 499.0047838361564;
            }
            case LIGHTMIN: {
                return this.distEM_ * 499.0047838361564 / 60.0;
            }
        }
        throw new IllegalArgumentException("No switch case for DistanceUnits");
    }

    public double getAlphaFMS() {
        return this.alphaFMS_;
    }

    public double getSolarLongitude() {
        return AbstractPlanetTime.normalize360(this.ellS_);
    }

    public double getSolarLongitudeAtPerihelion() {
        return AbstractPlanetTime.normalize360(this.ellSP_);
    }

    public PointLL getSubsolarPoint() {
        return new PointLL(this.getSubsolarLongitude(), this.getSolarDeclination());
    }

    public double getSolarDeclination() {
        return this.declinSPG_;
    }

    public double getSubsolarLongitude() {
        return this.subsolarLonPG_;
    }

    public Point2D.Double getSolarAngleAt(double lon, double lat) {
        return new Point2D.Double(this.getSolarAzimuthAngle(lon, lat), this.getSolarZenithAngle(lon, lat));
    }

    public double getSolarZenithAngle(double lon, double lat) {
        double cosDeclination = Math.cos(this.declinSPG_ * (Math.PI / 180));
        double sinDeclination = Math.sin(this.declinSPG_ * (Math.PI / 180));
        double hourAngle = lon - this.subsolarLonPG_;
        double cosHourAngle = Math.cos(hourAngle * (Math.PI / 180));
        double result = 57.29577951308232 * Math.acos(sinDeclination * Math.sin(lat * (Math.PI / 180)) + cosDeclination * Math.cos(lat * (Math.PI / 180)) * cosHourAngle);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.trace("xxx: hourAngle         = " + FORMAT_05.format(hourAngle));
            LOGGER.trace("D-5: solarZenithAngle  = " + FORMAT_05.format(result));
            LOGGER.trace("D-5: solarElevation    = " + FORMAT_05.format(90.0 - result));
        }
        return result;
    }

    public double getSolarAzimuthAngle(double lon, double lat) {
        double tanDec = Math.tan(this.declinSPG_ * (Math.PI / 180));
        double hourAngle = AbstractPlanetTime.normalize180(lon - this.subsolarLonPG_);
        double sinH = Math.sin(hourAngle * (Math.PI / 180));
        double cosH = Math.cos(hourAngle * (Math.PI / 180));
        double cosLat = Math.cos(lat * (Math.PI / 180));
        double sinLat = Math.sin(lat * (Math.PI / 180));
        double denominator = cosLat * tanDec - sinLat * cosH;
        double azimuthAngle = AbstractPlanetTime.normalize360(57.29577951308232 * Math.atan2(sinH, denominator));
        if (LOGGER.isDebugEnabled()) {
            LOGGER.trace("xxx: hourAngle         = " + FORMAT_05.format(hourAngle));
            LOGGER.trace("xxx: sinH              = " + FORMAT_05.format(sinH));
            LOGGER.trace("xxx: cosH              = " + FORMAT_05.format(cosH));
            LOGGER.trace("xxx: tanDec            = " + FORMAT_05.format(tanDec));
            LOGGER.trace("xxx: cosLat            = " + FORMAT_05.format(cosLat));
            LOGGER.trace("xxx: sinLat            = " + FORMAT_05.format(sinLat));
            LOGGER.trace("xxx: denominator       = " + FORMAT_05.format(denominator));
            LOGGER.trace("D-6: solarAzimuthAngle = " + FORMAT_05.format(azimuthAngle));
        }
        return azimuthAngle;
    }

    public PointLL getSubEarthPoint() {
        if (!this.gotEarthParams_) {
            this.calculateEarthParams();
        }
        return new PointLL(this.subEarthLonPG_, this.declinEPG_);
    }

    public Point2D.Double getEarthAngleAt(double lon, double lat) {
        PointLL earthAngle = this.getSubEarthPoint();
        double subLon = earthAngle.getLon();
        double subLatRad = earthAngle.getLat() * (Math.PI / 180);
        double cosDeclination = Math.cos(subLatRad);
        double sinDeclination = Math.sin(subLatRad);
        double tanDeclination = sinDeclination / cosDeclination;
        double hourAngle = AbstractPlanetTime.normalize180(lon - subLon);
        double cosLat = Math.cos(lat * (Math.PI / 180));
        double sinLat = Math.sin(lat * (Math.PI / 180));
        double cosHourAngle = Math.cos(hourAngle * (Math.PI / 180));
        double sinHourAngle = Math.sin(hourAngle * (Math.PI / 180));
        double azimuthAngle = AbstractPlanetTime.normalize360(57.29577951308232 * Math.atan2(sinHourAngle, cosLat * tanDeclination - sinLat * cosHourAngle));
        double zenithAngle = 57.29577951308232 * Math.acos(sinDeclination * sinLat + cosDeclination * cosLat * cosHourAngle);
        return new Point2D.Double(azimuthAngle, zenithAngle);
    }

    public double getTimeSolFraction(double lonPG, ClockType ctype) {
        double hours = this.getTimeHours(lonPG, ctype);
        return hours / 24.0;
    }

    public double getTimeHours(double lonPG, ClockType ctype) {
        if (ctype == ClockType.MEAN_SOLAR && lonPG == 0.0 || ctype == ClockType.PLANETARY) {
            return this.mst0_;
        }
        if (ctype == ClockType.TRUE_SOLAR && lonPG == 0.0) {
            return this.tst0_;
        }
        double xst = 0.0;
        if (ctype == ClockType.MEAN_ZONAL) {
            xst = this.mst0_ + (double)AbstractPlanetTime.getTimezoneOffset(lonPG);
        } else if (ctype == ClockType.MEAN_SOLAR) {
            xst = this.mst0_ - lonPG * 0.06666666666666667;
        } else if (ctype == ClockType.TRUE_SOLAR) {
            xst = this.tst0_ - lonPG * 0.06666666666666667;
        }
        xst = AbstractPlanetTime.normalize24(xst);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.trace("xxx: mst0              = " + FORMAT_06.format(this.mst0_));
            LOGGER.trace("xxx: tst0              = " + FORMAT_06.format(this.tst0_));
            LOGGER.trace("xxx: lonPG time offset = " + FORMAT_06.format(lonPG * 0.06666666666666667));
            LOGGER.trace("C-3: lmst              = " + FORMAT_06.format(this.mst0_ - lonPG * 0.06666666666666667));
            LOGGER.trace("C-4: ltst              = " + FORMAT_06.format(this.tst0_ - lonPG * 0.06666666666666667));
        }
        return xst;
    }

    public String getTimeStringMinutes(double lonPG, ClockType ctype, boolean appendSuffix) {
        return this.getTimeString(lonPG, ctype, TimeUnits.MINUTES, appendSuffix);
    }

    public String getTimeStringSeconds(double lonPG, ClockType ctype, boolean appendSuffix) {
        return this.getTimeString(lonPG, ctype, TimeUnits.SECONDS, appendSuffix);
    }

    public String getTimeString(double lonPG, ClockType ctype, TimeUnits smallest, boolean appendSuffix) {
        double hours = this.getTimeHours(lonPG, ctype);
        if (!appendSuffix) {
            return HourFormatter.format(hours, smallest, null, null);
        }
        switch (ctype) {
            case MEAN_SOLAR: {
                return HourFormatter.format(hours, smallest, null, "LMST");
            }
            case TRUE_SOLAR: {
                return HourFormatter.format(hours, smallest, null, "LTST");
            }
            case PLANETARY: {
                return HourFormatter.format(hours, smallest, null, this.pmTimeSuffix_);
            }
            case MEAN_ZONAL: {
                int tz = AbstractPlanetTime.getTimezoneOffset(lonPG);
                if (tz < 0) {
                    return HourFormatter.format(hours, smallest, null, this.pmTimeSuffix_ + tz);
                }
                return HourFormatter.format(hours, smallest, null, this.pmTimeSuffix_ + "+" + tz);
            }
        }
        throw new IllegalArgumentException("No switch case for ClockType");
    }

    public double getEOTHours() {
        return this.eotHours_;
    }

    public static int getTimezoneOffset(double lonPG) {
        return (int)Math.round(AbstractPlanetTime.normalize180(-lonPG) * 0.06666666666666667);
    }

    public String getSeason(boolean inNorth) {
        int seasonNo;
        StringBuilder theSeason = new StringBuilder(24);
        double ellS = AbstractPlanetTime.normalize360(this.ellS_);
        if (inNorth) {
            if (ellS >= 0.0 && ellS <= 0.5 || ellS >= 359.5 && ellS <= 360.0) {
                return "NH Vernal Equinox";
            }
            if (ellS >= 89.5 && ellS <= 90.5) {
                return "NH Summer Solstice";
            }
            if (ellS >= 179.5 && ellS <= 180.5) {
                return "NH Autumnal Equinox";
            }
            if (ellS >= 269.0 && ellS <= 270.5) {
                return "NH Winter Solstice";
            }
            seasonNo = (int)(ellS / 90.0);
            switch (seasonNo) {
                case 0: {
                    theSeason.append("NH Spring");
                    break;
                }
                case 1: {
                    theSeason.append("NH Summer");
                    break;
                }
                case 2: {
                    theSeason.append("NH Autumn");
                    break;
                }
                case 3: {
                    theSeason.append("NH Winter");
                    break;
                }
                default: {
                    throw new RuntimeException("Calculated season no. " + seasonNo);
                }
            }
        } else {
            if (ellS >= 0.0 && ellS <= 0.5 || ellS >= 359.5 && ellS <= 360.0) {
                return "SH Autumnal Equinox";
            }
            if (ellS >= 89.5 && ellS <= 90.5) {
                return "SH Winter Solstice";
            }
            if (ellS >= 179.5 && ellS <= 180.5) {
                return "SH Vernal Equinox";
            }
            if (ellS >= 269.0 && ellS <= 270.5) {
                return "SH Summer Solstice";
            }
            seasonNo = (int)(ellS / 90.0);
            switch (seasonNo) {
                case 0: {
                    theSeason.append("SH Autumn");
                    break;
                }
                case 1: {
                    theSeason.append("SH Winter");
                    break;
                }
                case 2: {
                    theSeason.append("SH Spring");
                    break;
                }
                case 3: {
                    theSeason.append("SH Summer");
                    break;
                }
                default: {
                    throw new RuntimeException("Calculated season no. " + seasonNo);
                }
            }
        }
        double section = ellS % 90.0;
        if (section < 30.0) {
            theSeason.insert(0, "Early ");
        } else if (section < 60.0) {
            theSeason.insert(0, "Mid ");
        } else {
            theSeason.insert(0, "Late ");
        }
        return theSeason.toString();
    }

    public double transformLat2PG(double latPC) {
        if (latPC > 90.0 || latPC < -90.0) {
            throw new IllegalArgumentException("Latitude out of range.");
        }
        if (this.flattening_ == 0.0 || latPC == 0.0 || latPC == 90.0 || latPC == -90.0) {
            return latPC;
        }
        return 57.29577951308232 * Math.atan(Math.tan(Math.PI / 180 * latPC) / (this.oneMinusFlatten_ * this.oneMinusFlatten_));
    }

    public double transformLat2PC(double latPG) {
        if (latPG > 90.0 || latPG < -90.0) {
            throw new IllegalArgumentException("Latitude out of range.");
        }
        if (this.flattening_ == 0.0 || latPG == 0.0 || latPG == 90.0 || latPG == -90.0) {
            return latPG;
        }
        return 57.29577951308232 * Math.atan(Math.tan(Math.PI / 180 * latPG) * this.oneMinusFlatten_ * this.oneMinusFlatten_);
    }

    protected static double normalize24(double d) {
        double dd;
        for (dd = d; dd < 0.0; dd += 24.0) {
        }
        while (dd >= 24.0) {
            dd -= 24.0;
        }
        return dd;
    }

    protected static double normalize180(double d) {
        double dd;
        for (dd = d; dd < -180.0; dd += 360.0) {
        }
        while (dd > 180.0) {
            dd -= 360.0;
        }
        return dd;
    }

    protected static double normalize360(double d) {
        double dd;
        for (dd = d; dd < 0.0; dd += 360.0) {
        }
        while (dd >= 360.0) {
            dd -= 360.0;
        }
        return dd;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addTimeListener(TimeListener l) {
        EventListenerList eventListenerList = this.listenerList_;
        synchronized (eventListenerList) {
            this.listenerList_.add(TimeListener.class, l);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeTimeListener(TimeListener l) {
        EventListenerList eventListenerList = this.listenerList_;
        synchronized (eventListenerList) {
            this.listenerList_.remove(TimeListener.class, l);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeTimeListeners() {
        EventListenerList eventListenerList = this.listenerList_;
        synchronized (eventListenerList) {
            TimeListener[] listeners = (TimeListener[])this.listenerList_.getListeners(TimeListener.class);
            for (int i = listeners.length - 1; i >= 0; --i) {
                this.removeTimeListener(listeners[i]);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void fireTimeChanged() {
        if (this.listenerList_ == null) {
            return;
        }
        TimeEvent e = new TimeEvent(this);
        EventListenerList eventListenerList = this.listenerList_;
        synchronized (eventListenerList) {
            TimeListener[] listeners = (TimeListener[])this.listenerList_.getListeners(TimeListener.class);
            for (int i = 0; i < listeners.length; ++i) {
                listeners[i].timeChanged(e);
            }
        }
    }
}

