/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.util.io;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.List;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jabref.logic.citationkeypattern.BracketedPattern;
import org.jabref.logic.util.io.FileFinder;
import org.jabref.logic.util.io.FileNameCleaner;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.strings.StringUtil;

class RegExpBasedFileFinder
implements FileFinder {
    private static final String EXT_MARKER = "__EXTENSION__";
    private static final Pattern ESCAPE_PATTERN = Pattern.compile("([^\\\\])\\\\([^\\\\])");
    private final String regExp;
    private final Character keywordDelimiter;

    RegExpBasedFileFinder(String regExp, Character keywordDelimiter) {
        this.regExp = regExp;
        this.keywordDelimiter = keywordDelimiter;
    }

    private Pattern createFileNamePattern(String[] fileParts, String extensionRegExp, BibEntry entry) throws IOException {
        String filePart = fileParts[fileParts.length - 1].replace("[extension]", EXT_MARKER);
        Function<String, String> expandBracket = BracketedPattern.expandBracketContent(this.keywordDelimiter, entry, null);
        Function<String, String> bracketToFileNameRegex = expandBracket.andThen(RegExpBasedFileFinder::toFileNameRegex);
        String expandedBracketAsFileNameRegex = BracketedPattern.expandBrackets(filePart, bracketToFileNameRegex);
        String fileNamePattern = expandedBracketAsFileNameRegex.replaceAll(EXT_MARKER, extensionRegExp).replaceAll("\\\\\\\\", "\\\\");
        try {
            return Pattern.compile("^" + fileNamePattern + "$", 2);
        }
        catch (PatternSyntaxException e) {
            throw new IOException("There is a syntax error in the regular expression %s used to search for files".formatted(fileNamePattern), e);
        }
    }

    private static String toFileNameRegex(String expandedContent) {
        String cleanedContent = FileNameCleaner.cleanFileName(expandedContent);
        return expandedContent.equals(cleanedContent) ? Pattern.quote(expandedContent) : "(" + Pattern.quote(expandedContent) + ")|(" + Pattern.quote(cleanedContent) + ")";
    }

    @Override
    public List<Path> findAssociatedFiles(BibEntry entry, List<Path> directories, List<String> extensions) throws IOException {
        String extensionRegExp = "(" + String.join((CharSequence)"|", extensions) + ")";
        return this.findFile(entry, directories, extensionRegExp);
    }

    private List<Path> findFile(BibEntry entry, List<Path> dirs, String extensionRegExp) throws IOException {
        ArrayList<Path> res = new ArrayList<Path>();
        for (Path directory : dirs) {
            res.addAll(this.findFile(entry, directory, this.regExp, extensionRegExp));
        }
        return res;
    }

    private List<Path> findFile(BibEntry entry, Path directory, String file, String extensionRegExp) throws IOException {
        Path actualDirectory;
        ArrayList<Path> resultFiles = new ArrayList<Path>();
        String fileName = file;
        if (fileName.startsWith("/")) {
            actualDirectory = Path.of(".", new String[0]);
            fileName = fileName.substring(1);
        } else {
            actualDirectory = directory;
        }
        Matcher m = ESCAPE_PATTERN.matcher(fileName);
        StringBuilder s = new StringBuilder();
        while (m.find()) {
            m.appendReplacement(s, m.group(1) + "/" + m.group(2));
        }
        m.appendTail(s);
        fileName = s.toString();
        String[] fileParts = fileName.split("/");
        if (fileParts.length == 0) {
            return resultFiles;
        }
        for (int index = 0; index < fileParts.length - 1; ++index) {
            File[] subDirs;
            String dirToProcess = fileParts[index];
            if (dirToProcess.matches("^.:$")) {
                actualDirectory = Path.of(dirToProcess + "/", new String[0]);
                continue;
            }
            if (".".equals(dirToProcess)) continue;
            if ("..".equals(dirToProcess)) {
                actualDirectory = actualDirectory.getParent();
                continue;
            }
            if ("*".equals(dirToProcess) && (subDirs = actualDirectory.toFile().listFiles()) != null) {
                String restOfFileString = StringUtil.join(fileParts, "/", index + 1, fileParts.length);
                for (File subDir : subDirs) {
                    if (!subDir.isDirectory()) continue;
                    resultFiles.addAll(this.findFile(entry, subDir.toPath(), restOfFileString, extensionRegExp));
                }
            }
            if (!"**".equals(dirToProcess)) continue;
            String restOfFileString = StringUtil.join(fileParts, "/", index + 1, fileParts.length);
            Path rootDirectory = actualDirectory;
            try (Stream<Path> pathStream = Files.walk(actualDirectory, new FileVisitOption[0]);){
                for (Path path2 : pathStream.filter(element -> this.isSubDirectory(rootDirectory, (Path)element)).collect(Collectors.toList())) {
                    resultFiles.addAll(this.findFile(entry, path2, restOfFileString, extensionRegExp));
                }
                continue;
            }
            catch (UncheckedIOException ioe) {
                throw ioe.getCause();
            }
        }
        Pattern toMatch = this.createFileNamePattern(fileParts, extensionRegExp, entry);
        BiPredicate<Path, BasicFileAttributes> matcher = (path, attributes) -> toMatch.matcher(path.getFileName().toString()).matches();
        try (Stream<Path> pathStream = Files.find(actualDirectory, 1, matcher, FileVisitOption.FOLLOW_LINKS);){
            resultFiles.addAll(pathStream.collect(Collectors.toList()));
        }
        catch (UncheckedIOException uncheckedIOException) {
            throw uncheckedIOException.getCause();
        }
        return resultFiles;
    }

    private boolean isSubDirectory(Path rootDirectory, Path path) {
        return !rootDirectory.equals(path) && Files.isDirectory(path, new LinkOption[0]);
    }
}

