/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.shared;

import com.google.common.collect.Lists;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import org.jabref.logic.shared.DBMSSynchronizer;
import org.jabref.logic.shared.DBMSType;
import org.jabref.logic.shared.DatabaseConnection;
import org.jabref.logic.shared.DatabaseConnectionProperties;
import org.jabref.logic.shared.MySQLProcessor;
import org.jabref.logic.shared.OracleProcessor;
import org.jabref.logic.shared.PostgreSQLProcessor;
import org.jabref.logic.shared.exception.OfflineLockException;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.SharedBibEntryData;
import org.jabref.model.entry.event.EntriesEventSource;
import org.jabref.model.entry.field.Field;
import org.jabref.model.entry.field.FieldFactory;
import org.jabref.model.entry.types.EntryTypeFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class DBMSProcessor {
    public static final String PROCESSOR_ID = UUID.randomUUID().toString();
    protected static final Logger LOGGER = LoggerFactory.getLogger(DBMSProcessor.class);
    protected final Connection connection;
    protected DatabaseConnectionProperties connectionProperties;

    protected DBMSProcessor(DatabaseConnection dbmsConnection) {
        this.connection = dbmsConnection.getConnection();
        this.connectionProperties = dbmsConnection.getProperties();
    }

    public boolean checkBaseIntegrity() throws SQLException {
        boolean databasePassesIntegrityCheck = false;
        DBMSType type = this.connectionProperties.getType();
        Map<String, String> metadata = this.getSharedMetaData();
        if (type == DBMSType.POSTGRESQL || type == DBMSType.MYSQL) {
            int VERSION_DB_STRUCT;
            String metadataVersion = metadata.get("VersionDBStructure");
            if (metadataVersion != null && (VERSION_DB_STRUCT = Integer.parseInt(metadata.getOrDefault("VersionDBStructure", "").replace(";", ""))) == this.getCURRENT_VERSION_DB_STRUCT()) {
                databasePassesIntegrityCheck = true;
            }
        } else {
            databasePassesIntegrityCheck = this.checkTableAvailability("ENTRY", "FIELD", "METADATA");
        }
        return databasePassesIntegrityCheck;
    }

    public boolean databaseIsAtMostJabRef35() throws SQLException {
        return this.checkTableAvailability("ENTRIES", "ENTRY_GROUP", "ENTRY_TYPES", "GROUPS", "GROUP_TYPES", "JABREF_DATABASE", "STRINGS");
    }

    protected boolean checkTableAvailability(String ... tableNames) throws SQLException {
        ArrayList<String> requiredTables = new ArrayList<String>();
        for (String name : tableNames) {
            requiredTables.add(name.toUpperCase(Locale.ENGLISH));
        }
        DatabaseMetaData databaseMetaData = this.connection.getMetaData();
        try (ResultSet databaseMetaDataResultSet = databaseMetaData.getTables(null, null, null, null);){
            while (databaseMetaDataResultSet.next()) {
                String tableName = databaseMetaDataResultSet.getString("TABLE_NAME").toUpperCase(Locale.ROOT);
                requiredTables.remove(tableName);
            }
            boolean bl = requiredTables.isEmpty();
            return bl;
        }
    }

    public void setupSharedDatabase() throws SQLException {
        this.setUp();
        if (!this.checkBaseIntegrity()) {
            LOGGER.error("Corrupt_shared_database_structure.");
        }
    }

    protected abstract void setUp() throws SQLException;

    abstract String escape(String var1);

    abstract String escape_Table(String var1);

    abstract Integer getCURRENT_VERSION_DB_STRUCT();

    public void insertEntry(BibEntry bibEntry) {
        this.insertEntries(Collections.singletonList(bibEntry));
    }

    public void insertEntries(List<BibEntry> bibEntries) {
        List<BibEntry> notYetExistingEntries = this.getNotYetExistingEntries(bibEntries);
        if (notYetExistingEntries.isEmpty()) {
            return;
        }
        this.insertIntoEntryTable(notYetExistingEntries);
        this.insertIntoFieldTable(notYetExistingEntries);
    }

    protected void insertIntoEntryTable(List<BibEntry> bibEntries) {
        StringBuilder insertIntoEntryQuery = new StringBuilder().append("INSERT INTO ").append(this.escape_Table("ENTRY")).append("(").append(this.escape("TYPE")).append(") VALUES(?)");
        insertIntoEntryQuery.append(", (?)".repeat(Math.max(0, bibEntries.size() - 1)));
        try (PreparedStatement preparedEntryStatement = this.connection.prepareStatement(insertIntoEntryQuery.toString(), new String[]{"SHARED_ID"});){
            for (int i = 0; i < bibEntries.size(); ++i) {
                preparedEntryStatement.setString(i + 1, bibEntries.get(i).getType().getName());
            }
            preparedEntryStatement.executeUpdate();
            try (ResultSet generatedKeys = preparedEntryStatement.getGeneratedKeys();){
                for (BibEntry bibEntry : bibEntries) {
                    generatedKeys.next();
                    bibEntry.getSharedBibEntryData().setSharedID(generatedKeys.getInt(1));
                }
                if (generatedKeys.next()) {
                    LOGGER.error("Error: Some shared IDs left unassigned");
                }
            }
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error: ", (Throwable)e);
        }
    }

    private List<BibEntry> getNotYetExistingEntries(List<BibEntry> bibEntries) {
        ArrayList<Integer> remoteIds = new ArrayList<Integer>();
        List<Integer> localIds = bibEntries.stream().map(BibEntry::getSharedBibEntryData).map(SharedBibEntryData::getSharedID).filter(id -> id != -1).toList();
        if (localIds.isEmpty()) {
            return bibEntries;
        }
        try {
            String selectQuery = "SELECT * FROM " + this.escape_Table("ENTRY");
            try (ResultSet resultSet = this.connection.createStatement().executeQuery(selectQuery);){
                while (resultSet.next()) {
                    int id2 = resultSet.getInt("SHARED_ID");
                    remoteIds.add(id2);
                }
            }
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error: ", (Throwable)e);
        }
        return bibEntries.stream().filter(entry -> !remoteIds.contains(entry.getSharedBibEntryData().getSharedID())).collect(Collectors.toList());
    }

    protected void insertIntoFieldTable(List<BibEntry> bibEntries) {
        try {
            List fields = bibEntries.stream().map(bibEntry -> new ArrayList<Field>(bibEntry.getFields())).collect(Collectors.toList());
            StringBuilder insertFieldQuery = new StringBuilder().append("INSERT INTO ").append(this.escape_Table("FIELD")).append("(").append(this.escape("ENTRY_SHARED_ID")).append(", ").append(this.escape("NAME")).append(", ").append(this.escape("VALUE")).append(") VALUES(?, ?, ?)");
            int numFields = 0;
            for (List entryFields : fields) {
                numFields += entryFields.size();
            }
            if (numFields == 0) {
                return;
            }
            insertFieldQuery.append(", (?, ?, ?)".repeat(Math.max(0, numFields - 1)));
            try (PreparedStatement preparedFieldStatement = this.connection.prepareStatement(insertFieldQuery.toString());){
                int fieldsCompleted = 0;
                for (int entryIndex = 0; entryIndex < fields.size(); ++entryIndex) {
                    for (int entryFieldsIndex = 0; entryFieldsIndex < ((List)fields.get(entryIndex)).size(); ++entryFieldsIndex) {
                        preparedFieldStatement.setInt(3 * fieldsCompleted + 1, bibEntries.get(entryIndex).getSharedBibEntryData().getSharedID());
                        preparedFieldStatement.setString(3 * fieldsCompleted + 2, ((Field)((List)fields.get(entryIndex)).get(entryFieldsIndex)).getName());
                        preparedFieldStatement.setString(3 * fieldsCompleted + 3, bibEntries.get(entryIndex).getField((Field)((List)fields.get(entryIndex)).get(entryFieldsIndex)).get());
                        ++fieldsCompleted;
                    }
                }
                preparedFieldStatement.executeUpdate();
            }
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error: ", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateEntry(BibEntry localBibEntry) throws OfflineLockException, SQLException {
        block15: {
            this.connection.setAutoCommit(false);
            try {
                Optional<BibEntry> sharedEntryOptional = this.getSharedEntry(localBibEntry.getSharedBibEntryData().getSharedID());
                if (sharedEntryOptional.isEmpty()) {
                    return;
                }
                BibEntry sharedBibEntry = sharedEntryOptional.get();
                this.removeSharedFieldsByDifference(localBibEntry, sharedBibEntry);
                if (localBibEntry.getSharedBibEntryData().getVersion() >= sharedBibEntry.getSharedBibEntryData().getVersion() || localBibEntry.equals(sharedBibEntry)) {
                    this.insertOrUpdateFields(localBibEntry);
                    String updateEntryTypeQuery = "UPDATE " + this.escape_Table("ENTRY") + " SET " + this.escape("TYPE") + " = ?, " + this.escape("VERSION") + " = " + this.escape("VERSION") + " + 1 WHERE " + this.escape("SHARED_ID") + " = ?";
                    try (PreparedStatement preparedUpdateEntryTypeStatement = this.connection.prepareStatement(updateEntryTypeQuery);){
                        preparedUpdateEntryTypeStatement.setString(1, localBibEntry.getType().getName());
                        preparedUpdateEntryTypeStatement.setInt(2, localBibEntry.getSharedBibEntryData().getSharedID());
                        preparedUpdateEntryTypeStatement.executeUpdate();
                    }
                    this.connection.commit();
                    break block15;
                }
                throw new OfflineLockException(localBibEntry, sharedBibEntry);
            }
            catch (SQLException e) {
                LOGGER.error("SQL Error: ", (Throwable)e);
                this.connection.rollback();
            }
            finally {
                this.connection.setAutoCommit(true);
            }
        }
    }

    private void removeSharedFieldsByDifference(BibEntry localBibEntry, BibEntry sharedBibEntry) throws SQLException {
        HashSet<Field> nullFields = new HashSet<Field>(sharedBibEntry.getFields());
        nullFields.removeAll(localBibEntry.getFields());
        for (Field nullField : nullFields) {
            String deleteFieldQuery = "DELETE FROM " + this.escape_Table("FIELD") + " WHERE " + this.escape("NAME") + " = ? AND " + this.escape("ENTRY_SHARED_ID") + " = ?";
            PreparedStatement preparedDeleteFieldStatement = this.connection.prepareStatement(deleteFieldQuery);
            try {
                preparedDeleteFieldStatement.setString(1, nullField.getName());
                preparedDeleteFieldStatement.setInt(2, localBibEntry.getSharedBibEntryData().getSharedID());
                preparedDeleteFieldStatement.executeUpdate();
            }
            finally {
                if (preparedDeleteFieldStatement == null) continue;
                preparedDeleteFieldStatement.close();
            }
        }
    }

    private void insertOrUpdateFields(BibEntry localBibEntry) throws SQLException {
        for (Field field : localBibEntry.getFields()) {
            Optional<String> valueOptional = localBibEntry.getField(field);
            String value = null;
            if (valueOptional.isPresent()) {
                value = valueOptional.get();
            }
            String selectFieldQuery = "SELECT * FROM " + this.escape_Table("FIELD") + " WHERE " + this.escape("NAME") + " = ? AND " + this.escape("ENTRY_SHARED_ID") + " = ?";
            PreparedStatement preparedSelectFieldStatement = this.connection.prepareStatement(selectFieldQuery);
            try {
                preparedSelectFieldStatement.setString(1, field.getName());
                preparedSelectFieldStatement.setInt(2, localBibEntry.getSharedBibEntryData().getSharedID());
                ResultSet selectFieldResultSet = preparedSelectFieldStatement.executeQuery();
                try {
                    if (selectFieldResultSet.next()) {
                        String updateFieldQuery = "UPDATE " + this.escape_Table("FIELD") + " SET " + this.escape("VALUE") + " = ? WHERE " + this.escape("NAME") + " = ? AND " + this.escape("ENTRY_SHARED_ID") + " = ?";
                        PreparedStatement preparedUpdateFieldStatement = this.connection.prepareStatement(updateFieldQuery);
                        try {
                            preparedUpdateFieldStatement.setString(1, value);
                            preparedUpdateFieldStatement.setString(2, field.getName());
                            preparedUpdateFieldStatement.setInt(3, localBibEntry.getSharedBibEntryData().getSharedID());
                            preparedUpdateFieldStatement.executeUpdate();
                            continue;
                        }
                        finally {
                            if (preparedUpdateFieldStatement != null) {
                                preparedUpdateFieldStatement.close();
                            }
                            continue;
                        }
                    }
                    String insertFieldQuery = "INSERT INTO " + this.escape_Table("FIELD") + "(" + this.escape("ENTRY_SHARED_ID") + ", " + this.escape("NAME") + ", " + this.escape("VALUE") + ") VALUES(?, ?, ?)";
                    PreparedStatement preparedFieldStatement = this.connection.prepareStatement(insertFieldQuery);
                    try {
                        preparedFieldStatement.setInt(1, localBibEntry.getSharedBibEntryData().getSharedID());
                        preparedFieldStatement.setString(2, field.getName());
                        preparedFieldStatement.setString(3, value);
                        preparedFieldStatement.executeUpdate();
                    }
                    finally {
                        if (preparedFieldStatement == null) continue;
                        preparedFieldStatement.close();
                    }
                }
                finally {
                    if (selectFieldResultSet == null) continue;
                    selectFieldResultSet.close();
                }
            }
            finally {
                if (preparedSelectFieldStatement == null) continue;
                preparedSelectFieldStatement.close();
            }
        }
    }

    public void removeEntries(List<BibEntry> bibEntries) {
        Objects.requireNonNull(bibEntries);
        if (bibEntries.isEmpty()) {
            return;
        }
        StringBuilder query = new StringBuilder().append("DELETE FROM ").append(this.escape_Table("ENTRY")).append(" WHERE ").append(this.escape("SHARED_ID")).append(" IN (");
        query.append("?, ".repeat(bibEntries.size() - 1));
        query.append("?)");
        try (PreparedStatement preparedStatement = this.connection.prepareStatement(query.toString());){
            for (int j = 0; j < bibEntries.size(); ++j) {
                preparedStatement.setInt(j + 1, bibEntries.get(j).getSharedBibEntryData().getSharedID());
            }
            preparedStatement.executeUpdate();
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error: ", (Throwable)e);
        }
    }

    public Optional<BibEntry> getSharedEntry(int sharedID) {
        List<BibEntry> sharedEntries = this.getSharedEntries(Collections.singletonList(sharedID));
        if (sharedEntries.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(sharedEntries.getFirst());
    }

    public List<BibEntry> partitionAndGetSharedEntries(List<Integer> sharedIDs) {
        List partitions = Lists.partition(sharedIDs, (int)500);
        ArrayList<BibEntry> result = new ArrayList<BibEntry>();
        for (List sublist : partitions) {
            result.addAll(this.getSharedEntries(sublist));
        }
        return result;
    }

    public List<BibEntry> getSharedEntries(List<Integer> sharedIDs) {
        Objects.requireNonNull(sharedIDs);
        ArrayList<BibEntry> sharedEntries = new ArrayList<BibEntry>();
        StringBuilder query = new StringBuilder();
        query.append("SELECT ").append(this.escape_Table("ENTRY")).append(".").append(this.escape("SHARED_ID")).append(", ").append(this.escape_Table("ENTRY")).append(".").append(this.escape("TYPE")).append(", ").append(this.escape_Table("ENTRY")).append(".").append(this.escape("VERSION")).append(", ").append("F.").append(this.escape("ENTRY_SHARED_ID")).append(", ").append("F.").append(this.escape("NAME")).append(", ").append("F.").append(this.escape("VALUE")).append(" FROM ").append(this.escape_Table("ENTRY")).append(" left outer join ").append(this.escape_Table("FIELD")).append(" F on ").append(this.escape_Table("ENTRY")).append(".").append(this.escape("SHARED_ID")).append(" = F.").append(this.escape("ENTRY_SHARED_ID"));
        if (!sharedIDs.isEmpty()) {
            query.append(" where ").append(this.escape("SHARED_ID")).append(" in (").append("?, ".repeat(sharedIDs.size() - 1)).append("?)");
        }
        query.append(" order by ").append(this.escape("SHARED_ID"));
        try (PreparedStatement preparedStatement = this.connection.prepareStatement(query.toString());){
            for (int i = 0; i < sharedIDs.size(); ++i) {
                preparedStatement.setInt(i + 1, sharedIDs.get(i));
            }
            try (ResultSet selectEntryResultSet = preparedStatement.executeQuery();){
                BibEntry bibEntry = null;
                int lastId = -1;
                while (selectEntryResultSet.next()) {
                    String value;
                    if (selectEntryResultSet.getInt("SHARED_ID") > lastId) {
                        bibEntry = new BibEntry();
                        bibEntry.getSharedBibEntryData().setSharedID(selectEntryResultSet.getInt("SHARED_ID"));
                        bibEntry.setType(EntryTypeFactory.parse(selectEntryResultSet.getString("TYPE")));
                        bibEntry.getSharedBibEntryData().setVersion(selectEntryResultSet.getInt("VERSION"));
                        sharedEntries.add(bibEntry);
                        lastId = selectEntryResultSet.getInt("SHARED_ID");
                    }
                    if ((value = selectEntryResultSet.getString("VALUE")) == null || bibEntry == null) continue;
                    bibEntry.setField(FieldFactory.parseField(selectEntryResultSet.getString("NAME")), value, EntriesEventSource.SHARED);
                }
            }
        }
        catch (SQLException e) {
            LOGGER.error("Executed >{}<", (Object)query);
            LOGGER.error("SQL Error", (Throwable)e);
            return Collections.emptyList();
        }
        return sharedEntries;
    }

    public List<BibEntry> getSharedEntries() {
        return this.getSharedEntries(Collections.emptyList());
    }

    public Map<Integer, Integer> getSharedIDVersionMapping() {
        HashMap<Integer, Integer> sharedIDVersionMapping = new HashMap<Integer, Integer>();
        String selectEntryQuery = "SELECT * FROM " + this.escape_Table("ENTRY") + " ORDER BY " + this.escape("SHARED_ID");
        try (ResultSet selectEntryResultSet = this.connection.createStatement().executeQuery(selectEntryQuery);){
            while (selectEntryResultSet.next()) {
                sharedIDVersionMapping.put(selectEntryResultSet.getInt("SHARED_ID"), selectEntryResultSet.getInt("VERSION"));
            }
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error", (Throwable)e);
        }
        return sharedIDVersionMapping;
    }

    public Map<String, String> getSharedMetaData() {
        HashMap<String, String> data = new HashMap<String, String>();
        try (ResultSet resultSet = this.connection.createStatement().executeQuery("SELECT * FROM " + this.escape_Table("METADATA"));){
            while (resultSet.next()) {
                data.put(resultSet.getString("KEY"), resultSet.getString("VALUE"));
            }
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error", (Throwable)e);
        }
        return data;
    }

    public void setSharedMetaData(Map<String, String> data) throws SQLException {
        StringBuilder updateQuery = new StringBuilder().append("UPDATE ").append(this.escape_Table("METADATA")).append(" SET ").append(this.escape("VALUE")).append(" = ? ").append(" WHERE ").append(this.escape("KEY")).append(" = ?");
        StringBuilder insertQuery = new StringBuilder().append("INSERT INTO ").append(this.escape_Table("METADATA")).append("(").append(this.escape("KEY")).append(", ").append(this.escape("VALUE")).append(") VALUES(?, ?)");
        for (Map.Entry<String, String> metaEntry : data.entrySet()) {
            try {
                PreparedStatement updateStatement = this.connection.prepareStatement(updateQuery.toString());
                try {
                    updateStatement.setString(2, metaEntry.getKey());
                    updateStatement.setString(1, metaEntry.getValue());
                    if (updateStatement.executeUpdate() != 0) continue;
                    try {
                        PreparedStatement insertStatement = this.connection.prepareStatement(insertQuery.toString());
                        try {
                            insertStatement.setString(1, metaEntry.getKey());
                            insertStatement.setString(2, metaEntry.getValue());
                            insertStatement.executeUpdate();
                        }
                        finally {
                            if (insertStatement == null) continue;
                            insertStatement.close();
                        }
                    }
                    catch (SQLException e) {
                        LOGGER.error("SQL Error: ", (Throwable)e);
                    }
                }
                finally {
                    if (updateStatement == null) continue;
                    updateStatement.close();
                }
            }
            catch (SQLException e) {
                LOGGER.error("SQL Error: ", (Throwable)e);
            }
        }
    }

    public static DBMSProcessor getProcessorInstance(DatabaseConnection connection) {
        DBMSType type = connection.getProperties().getType();
        if (type == DBMSType.MYSQL) {
            return new MySQLProcessor(connection);
        }
        if (type == DBMSType.POSTGRESQL) {
            return new PostgreSQLProcessor(connection);
        }
        if (type == DBMSType.ORACLE) {
            return new OracleProcessor(connection);
        }
        return null;
    }

    public DatabaseConnectionProperties getDBMSConnectionProperties() {
        return this.connectionProperties;
    }

    public void startNotificationListener(DBMSSynchronizer dbmsSynchronizer) {
    }

    public void stopNotificationListener() {
    }

    public void notifyClients() {
    }
}

