/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.pdf.search;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StringField;
import org.apache.lucene.document.TextField;
import org.apache.lucene.index.IndexableField;
import org.apache.pdfbox.Loader;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.interactive.annotation.PDAnnotation;
import org.apache.pdfbox.text.PDFTextStripper;
import org.jabref.gui.LibraryTab;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.LinkedFile;
import org.jabref.model.strings.StringUtil;
import org.jabref.preferences.FilePreferences;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DocumentReader {
    private static final Logger LOGGER = LoggerFactory.getLogger(LibraryTab.class);
    private static final Pattern HYPHEN_LINEBREAK_PATTERN = Pattern.compile("\\-\n");
    private static final Pattern LINEBREAK_WITHOUT_PERIOD_PATTERN = Pattern.compile("([^\\\\.])\\n");
    private final BibEntry entry;
    private final FilePreferences filePreferences;

    public DocumentReader(BibEntry bibEntry, FilePreferences filePreferences) {
        this.filePreferences = filePreferences;
        if (bibEntry.getFiles().isEmpty()) {
            throw new IllegalStateException("There are no linked PDF files to this BibEntry.");
        }
        this.entry = bibEntry;
    }

    public Optional<List<Document>> readLinkedPdf(BibDatabaseContext databaseContext, LinkedFile pdf) {
        Optional<Path> pdfPath = pdf.findIn(databaseContext, this.filePreferences);
        if (pdfPath.isPresent()) {
            return Optional.of(this.readPdfContents(pdf, pdfPath.get()));
        }
        return Optional.empty();
    }

    public List<Document> readLinkedPdfs(BibDatabaseContext databaseContext) {
        return this.entry.getFiles().stream().map(pdf -> this.readLinkedPdf(databaseContext, (LinkedFile)pdf)).filter(Optional::isPresent).map(Optional::get).flatMap(Collection::stream).collect(Collectors.toList());
    }

    private List<Document> readPdfContents(LinkedFile pdf, Path resolvedPdfPath) {
        ArrayList<Document> pages = new ArrayList<Document>();
        try (PDDocument pdfDocument = Loader.loadPDF((File)resolvedPdfPath.toFile());){
            for (int pageNumber = 0; pageNumber < pdfDocument.getNumberOfPages(); ++pageNumber) {
                Document newDocument = new Document();
                this.addIdentifiers(newDocument, pdf.getLink());
                this.addMetaData(newDocument, resolvedPdfPath, pageNumber);
                try {
                    this.addContentIfNotEmpty(pdfDocument, newDocument, pageNumber);
                }
                catch (IOException e) {
                    LOGGER.warn("Could not read page {} of  {}", new Object[]{pageNumber, resolvedPdfPath.toAbsolutePath(), e});
                }
                pages.add(newDocument);
            }
        }
        catch (IOException e) {
            LOGGER.warn("Could not read {}", (Object)resolvedPdfPath.toAbsolutePath(), (Object)e);
        }
        if (pages.isEmpty()) {
            Document newDocument = new Document();
            this.addIdentifiers(newDocument, pdf.getLink());
            this.addMetaData(newDocument, resolvedPdfPath, 0);
            pages.add(newDocument);
        }
        return pages;
    }

    private void addMetaData(Document newDocument, Path resolvedPdfPath, int pageNumber) {
        try {
            BasicFileAttributes attributes = Files.readAttributes(resolvedPdfPath, BasicFileAttributes.class, new LinkOption[0]);
            this.addStringField(newDocument, "modified", String.valueOf(attributes.lastModifiedTime().to(TimeUnit.SECONDS)));
        }
        catch (IOException e) {
            LOGGER.error("Could not read timestamp for {}", (Object)resolvedPdfPath, (Object)e);
        }
        this.addStringField(newDocument, "pageNumber", String.valueOf(pageNumber));
    }

    private void addStringField(Document newDocument, String field, String value) {
        if (!this.isValidField(value)) {
            return;
        }
        newDocument.add((IndexableField)new StringField(field, value, Field.Store.YES));
    }

    private boolean isValidField(String value) {
        return !StringUtil.isNullOrEmpty(value);
    }

    public static String mergeLines(String text) {
        String mergedHyphenNewlines = HYPHEN_LINEBREAK_PATTERN.matcher(text).replaceAll("");
        return LINEBREAK_WITHOUT_PERIOD_PATTERN.matcher(mergedHyphenNewlines).replaceAll("$1 ");
    }

    private void addContentIfNotEmpty(PDDocument pdfDocument, Document newDocument, int pageNumber) throws IOException {
        PDPage page;
        List annotations;
        PDFTextStripper pdfTextStripper = new PDFTextStripper();
        pdfTextStripper.setLineSeparator("\n");
        pdfTextStripper.setStartPage(pageNumber + 1);
        pdfTextStripper.setEndPage(pageNumber + 1);
        String pdfContent = pdfTextStripper.getText(pdfDocument);
        if (StringUtil.isNotBlank(pdfContent)) {
            newDocument.add((IndexableField)new TextField("content", DocumentReader.mergeLines(pdfContent), Field.Store.YES));
        }
        if (!(annotations = (page = pdfDocument.getPage(pageNumber)).getAnnotations().stream().filter(annotation -> annotation.getContents() != null).map(PDAnnotation::getContents).collect(Collectors.toList())).isEmpty()) {
            newDocument.add((IndexableField)new TextField("annotations", annotations.stream().collect(Collectors.joining("\n")), Field.Store.YES));
        }
    }

    private void addIdentifiers(Document newDocument, String path) {
        newDocument.add((IndexableField)new StringField("path", path, Field.Store.YES));
    }
}

