/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.importer.fetcher;

import com.google.common.util.concurrent.RateLimiter;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.regex.Pattern;
import kong.unirest.core.json.JSONArray;
import kong.unirest.core.json.JSONException;
import kong.unirest.core.json.JSONObject;
import org.jabref.logic.cleanup.FieldFormatterCleanup;
import org.jabref.logic.formatter.bibtexfields.ClearFormatter;
import org.jabref.logic.formatter.bibtexfields.HtmlToLatexFormatter;
import org.jabref.logic.formatter.bibtexfields.NormalizePagesFormatter;
import org.jabref.logic.help.HelpFile;
import org.jabref.logic.importer.EntryBasedFetcher;
import org.jabref.logic.importer.FetcherException;
import org.jabref.logic.importer.IdBasedFetcher;
import org.jabref.logic.importer.ImportFormatPreferences;
import org.jabref.logic.importer.ParseException;
import org.jabref.logic.importer.fetcher.Medra;
import org.jabref.logic.importer.fileformat.BibtexParser;
import org.jabref.logic.l10n.Localization;
import org.jabref.logic.net.URLDownload;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.field.StandardField;
import org.jabref.model.entry.identifier.DOI;
import org.jabref.model.entry.types.StandardEntryType;
import org.jabref.model.util.OptionalUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DoiFetcher
implements IdBasedFetcher,
EntryBasedFetcher {
    public static final String NAME = "DOI";
    private static final String APS_JOURNAL_ORG_DOI_ID = "1103";
    private static final String APS_SUFFIX = "([\\w]+\\.)([\\w]+\\.)([\\w]+)";
    private static final Pattern APS_SUFFIX_PATTERN = Pattern.compile("([\\w]+\\.)([\\w]+\\.)([\\w]+)");
    private static final Logger LOGGER = LoggerFactory.getLogger(DoiFetcher.class);
    private static final RateLimiter DATA_CITE_DCN_RATE_LIMITER = RateLimiter.create((double)3.33);
    private static final RateLimiter CROSSREF_DCN_RATE_LIMITER = RateLimiter.create((double)50.0);
    private final ImportFormatPreferences preferences;

    public DoiFetcher(ImportFormatPreferences preferences) {
        this.preferences = preferences;
    }

    @Override
    public String getName() {
        return NAME;
    }

    @Override
    public Optional<HelpFile> getHelpPage() {
        return Optional.of(HelpFile.FETCHER_DOI);
    }

    private void doAPILimiting(String identifier) {
        Optional<DOI> doi = DOI.parse(identifier);
        try {
            Optional<String> agency;
            if (doi.isPresent() && (agency = this.getAgency(doi.get())).isPresent()) {
                double waitingTime = 0.0;
                if ("datacite".equalsIgnoreCase(agency.get())) {
                    waitingTime = DATA_CITE_DCN_RATE_LIMITER.acquire();
                } else if ("crossref".equalsIgnoreCase(agency.get())) {
                    waitingTime = CROSSREF_DCN_RATE_LIMITER.acquire();
                }
                LOGGER.trace("Thread %s, searching for DOI '%s', waited %.2fs because of API rate limiter".formatted(Thread.currentThread().threadId(), identifier, waitingTime));
            }
        }
        catch (IOException e) {
            LOGGER.warn("Could not limit DOI API access rate", (Throwable)e);
        }
    }

    protected CompletableFuture<Optional<BibEntry>> asyncPerformSearchById(String identifier) {
        this.doAPILimiting(identifier);
        return CompletableFuture.supplyAsync(() -> {
            try {
                return this.performSearchById(identifier);
            }
            catch (FetcherException e) {
                throw new CompletionException(e);
            }
        });
    }

    @Override
    public Optional<BibEntry> performSearchById(String identifier) throws FetcherException {
        Optional<DOI> doi = DOI.parse(identifier);
        try {
            if (doi.isPresent()) {
                BibEntry entry;
                String bibtexString;
                URLConnection openConnection;
                Optional<String> agency = this.getAgency(doi.get());
                if (agency.isPresent() && "medra".equalsIgnoreCase(agency.get())) {
                    return new Medra().performSearchById(identifier);
                }
                URL doiURL = new URL(doi.get().getURIAsASCIIString());
                URLDownload download = this.getUrlDownload(doiURL);
                download.addHeader("Accept", "application/x-bibtex");
                try {
                    openConnection = download.openConnection();
                    bibtexString = URLDownload.asString(openConnection).trim();
                }
                catch (IOException e) {
                    Throwable throwable = e.getCause();
                    if (throwable instanceof FetcherException) {
                        FetcherException fe = (FetcherException)throwable;
                        throw fe;
                    }
                    throw e;
                }
                Optional<BibEntry> fetchedEntry = BibtexParser.singleFromString(bibtexString, this.preferences);
                fetchedEntry.ifPresent(this::doPostCleanup);
                if (agency.isPresent() && "crossref".equalsIgnoreCase(agency.get())) {
                    this.updateCrossrefAPIRate(openConnection);
                }
                if (fetchedEntry.isPresent() && fetchedEntry.get().hasField(StandardField.DOI) && this.isAPSJournal(entry = fetchedEntry.get(), entry.getField(StandardField.DOI).get()) && !entry.hasField(StandardField.PAGES)) {
                    this.setPageCountToArticleId(entry, entry.getField(StandardField.DOI).get());
                }
                if (openConnection instanceof HttpURLConnection) {
                    HttpURLConnection connection = (HttpURLConnection)openConnection;
                    connection.disconnect();
                }
                return fetchedEntry;
            }
            throw new FetcherException(Localization.lang("Invalid DOI: '%0'.", identifier));
        }
        catch (IOException e) {
            throw new FetcherException(Localization.lang("Connection error", new Object[0]), e);
        }
        catch (ParseException e) {
            throw new FetcherException("Could not parse BibTeX entry", e);
        }
        catch (JSONException e) {
            throw new FetcherException("Could not retrieve Registration Agency", e);
        }
    }

    private void doPostCleanup(BibEntry entry) {
        new FieldFormatterCleanup(StandardField.PAGES, new NormalizePagesFormatter()).cleanup(entry);
        new FieldFormatterCleanup(StandardField.URL, new ClearFormatter()).cleanup(entry);
        new FieldFormatterCleanup(StandardField.TITLE, new HtmlToLatexFormatter()).cleanup(entry);
    }

    private void updateCrossrefAPIRate(URLConnection existingConnection) {
        try {
            String xRateLimitInterval = existingConnection.getHeaderField("X-Rate-Limit-Interval").replaceAll("[^\\.0123456789]", "");
            String xRateLimit = existingConnection.getHeaderField("X-Rate-Limit-Limit");
            double newRate = Double.parseDouble(xRateLimit) / Double.parseDouble(xRateLimitInterval);
            double oldRate = CROSSREF_DCN_RATE_LIMITER.getRate();
            if (Math.abs(newRate - oldRate) >= 1.0) {
                LOGGER.info("Updated Crossref API rate limit from %.2f to %.2f".formatted(oldRate, newRate));
                CROSSREF_DCN_RATE_LIMITER.setRate(newRate);
            }
        }
        catch (IllegalArgumentException | NullPointerException e) {
            LOGGER.warn("Could not deduce Crossref API's rate limit from response header. API might have changed");
        }
    }

    @Override
    public List<BibEntry> performSearch(BibEntry entry) throws FetcherException {
        Optional<String> doi = entry.getField(StandardField.DOI);
        if (doi.isPresent()) {
            return OptionalUtil.toList(this.performSearchById(doi.get()));
        }
        return Collections.emptyList();
    }

    public Optional<String> getAgency(DOI doi) throws IOException {
        Optional<String> agency = Optional.empty();
        try {
            URLDownload download = this.getUrlDownload(new URL(String.valueOf(DOI.AGENCY_RESOLVER) + "/" + doi.getDOI()));
            JSONObject response = new JSONArray(download.asString()).getJSONObject(0);
            if (response != null) {
                agency = Optional.ofNullable(response.optString("RA"));
            }
        }
        catch (JSONException e) {
            LOGGER.error("Cannot parse agency fetcher response to JSON");
            return Optional.empty();
        }
        return agency;
    }

    private void setPageCountToArticleId(BibEntry entry, String doiAsString) {
        String articleId = doiAsString.substring(doiAsString.lastIndexOf(46) + 1);
        entry.setField(StandardField.PAGES, articleId);
    }

    private boolean isAPSJournal(BibEntry entry, String doiAsString) {
        if (!entry.getType().equals(StandardEntryType.Article)) {
            return false;
        }
        String suffix = doiAsString.substring(doiAsString.lastIndexOf(47) + 1);
        String organizationId = doiAsString.substring(doiAsString.indexOf(46) + 1, doiAsString.indexOf(47));
        return organizationId.equals(APS_JOURNAL_ORG_DOI_ID) && APS_SUFFIX_PATTERN.matcher(suffix).matches();
    }
}

