/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.crawler;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.charset.UnsupportedCharsetException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.jabref.logic.citationkeypattern.CitationKeyGenerator;
import org.jabref.logic.crawler.StudyCatalogToFetcherConverter;
import org.jabref.logic.crawler.StudyYamlParser;
import org.jabref.logic.database.DatabaseMerger;
import org.jabref.logic.exporter.AtomicFileWriter;
import org.jabref.logic.exporter.BibWriter;
import org.jabref.logic.exporter.BibtexDatabaseWriter;
import org.jabref.logic.exporter.SaveException;
import org.jabref.logic.exporter.SelfContainedSaveConfiguration;
import org.jabref.logic.git.SlrGitHandler;
import org.jabref.logic.importer.OpenDatabase;
import org.jabref.logic.importer.SearchBasedFetcher;
import org.jabref.logic.l10n.Localization;
import org.jabref.logic.util.OS;
import org.jabref.logic.util.io.FileNameCleaner;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.entry.BibEntryTypesManager;
import org.jabref.model.metadata.SaveOrder;
import org.jabref.model.metadata.SelfContainedSaveOrder;
import org.jabref.model.study.FetchResult;
import org.jabref.model.study.QueryResult;
import org.jabref.model.study.Study;
import org.jabref.model.study.StudyDatabase;
import org.jabref.model.study.StudyQuery;
import org.jabref.model.util.FileUpdateMonitor;
import org.jabref.preferences.PreferencesService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StudyRepository {
    public static final String STUDY_DEFINITION_FILE_NAME = "study.yml";
    private static final Logger LOGGER = LoggerFactory.getLogger(StudyRepository.class);
    private static final Pattern MATCH_COLON = Pattern.compile(":");
    private static final Pattern MATCH_ILLEGAL_CHARACTERS = Pattern.compile("[^A-Za-z0-9_.\\s=-]");
    private static final String REMOTE = "origin";
    private static final String WORK_BRANCH = "work";
    private static final String SEARCH_BRANCH = "search";
    private final Path repositoryPath;
    private final Path studyDefinitionFile;
    private final SlrGitHandler gitHandler;
    private final Study study;
    private final PreferencesService preferencesService;
    private final FileUpdateMonitor fileUpdateMonitor;
    private final BibEntryTypesManager bibEntryTypesManager;

    public StudyRepository(Path pathToRepository, SlrGitHandler gitHandler, PreferencesService preferencesService, FileUpdateMonitor fileUpdateMonitor, BibEntryTypesManager bibEntryTypesManager) throws IOException {
        this.repositoryPath = pathToRepository;
        this.gitHandler = gitHandler;
        this.preferencesService = preferencesService;
        this.fileUpdateMonitor = fileUpdateMonitor;
        this.studyDefinitionFile = Path.of(this.repositoryPath.toString(), STUDY_DEFINITION_FILE_NAME);
        this.bibEntryTypesManager = bibEntryTypesManager;
        if (Files.notExists(this.repositoryPath, new LinkOption[0])) {
            throw new IOException("The given repository does not exists.");
        }
        try {
            gitHandler.createCommitOnCurrentBranch("Save changes before searching.", false);
            gitHandler.checkoutBranch(WORK_BRANCH);
            this.updateWorkAndSearchBranch();
        }
        catch (GitAPIException e) {
            LOGGER.error("Could not checkout work branch");
        }
        if (Files.notExists(this.studyDefinitionFile, new LinkOption[0])) {
            throw new IOException("The study definition file does not exist in the given repository.");
        }
        this.study = this.parseStudyFile();
        try {
            boolean studyDefinitionDoesNotExistOrChanged;
            String updateRepositoryStructureMessage = "Update repository structure";
            this.setUpRepositoryStructureForQueriesAndFetchers();
            gitHandler.createCommitOnCurrentBranch("Update repository structure", false);
            gitHandler.checkoutBranch(SEARCH_BRANCH);
            boolean bl = studyDefinitionDoesNotExistOrChanged = !Files.exists(this.studyDefinitionFile, new LinkOption[0]) || !new StudyYamlParser().parseStudyYamlFile(this.studyDefinitionFile).equals(this.study);
            if (studyDefinitionDoesNotExistOrChanged) {
                new StudyYamlParser().writeStudyYamlFile(this.study, this.studyDefinitionFile);
            }
            this.setUpRepositoryStructureForQueriesAndFetchers();
            gitHandler.createCommitOnCurrentBranch("Update repository structure", false);
        }
        catch (GitAPIException e) {
            LOGGER.error("Could not checkout search branch.");
        }
        try {
            gitHandler.checkoutBranch(WORK_BRANCH);
        }
        catch (GitAPIException e) {
            LOGGER.error("Could not checkout work branch");
        }
    }

    public BibDatabaseContext getFetcherResultEntries(String query, String fetcherName) throws IOException {
        if (Files.exists(this.getPathToFetcherResultFile(query, fetcherName), new LinkOption[0])) {
            return OpenDatabase.loadDatabase(this.getPathToFetcherResultFile(query, fetcherName), this.preferencesService.getImportFormatPreferences(), this.fileUpdateMonitor).getDatabaseContext();
        }
        return new BibDatabaseContext();
    }

    public BibDatabaseContext getQueryResultEntries(String query) throws IOException {
        if (Files.exists(this.getPathToQueryResultFile(query), new LinkOption[0])) {
            return OpenDatabase.loadDatabase(this.getPathToQueryResultFile(query), this.preferencesService.getImportFormatPreferences(), this.fileUpdateMonitor).getDatabaseContext();
        }
        return new BibDatabaseContext();
    }

    public BibDatabaseContext getStudyResultEntries() throws IOException {
        if (Files.exists(this.getPathToStudyResultFile(), new LinkOption[0])) {
            return OpenDatabase.loadDatabase(this.getPathToStudyResultFile(), this.preferencesService.getImportFormatPreferences(), this.fileUpdateMonitor).getDatabaseContext();
        }
        return new BibDatabaseContext();
    }

    private Study parseStudyFile() throws IOException {
        return new StudyYamlParser().parseStudyYamlFile(this.studyDefinitionFile);
    }

    public List<String> getSearchQueryStrings() {
        return this.study.getQueries().parallelStream().map(StudyQuery::getQuery).collect(Collectors.toList());
    }

    public List<StudyDatabase> getActiveLibraryEntries() throws IllegalArgumentException {
        return this.study.getDatabases().parallelStream().filter(StudyDatabase::isEnabled).collect(Collectors.toList());
    }

    public Study getStudy() {
        return this.study;
    }

    public void persist(List<QueryResult> crawlResults) throws IOException, GitAPIException, SaveException {
        this.updateWorkAndSearchBranch();
        this.gitHandler.checkoutBranch(SEARCH_BRANCH);
        this.persistResults(crawlResults);
        try {
            String commitMessage = "Conducted search: " + String.valueOf(LocalDateTime.now().truncatedTo(ChronoUnit.SECONDS));
            boolean newSearchResults = this.gitHandler.createCommitOnCurrentBranch(commitMessage, false);
            this.gitHandler.checkoutBranch(WORK_BRANCH);
            if (!newSearchResults) {
                return;
            }
            this.gitHandler.appendLatestSearchResultsOntoCurrentBranch(commitMessage + " - Patch", SEARCH_BRANCH);
            this.updateRemoteSearchAndWorkBranch();
        }
        catch (GitAPIException e) {
            LOGGER.error("Updating remote repository failed", (Throwable)e);
        }
    }

    private void updateRemoteSearchAndWorkBranch() throws IOException, GitAPIException {
        String currentBranch = this.gitHandler.getCurrentlyCheckedOutBranch();
        this.gitHandler.checkoutBranch(SEARCH_BRANCH);
        this.gitHandler.pushCommitsToRemoteRepository();
        this.gitHandler.checkoutBranch(WORK_BRANCH);
        this.gitHandler.pushCommitsToRemoteRepository();
        this.gitHandler.checkoutBranch(currentBranch);
    }

    private void updateWorkAndSearchBranch() throws IOException, GitAPIException {
        String currentBranch = this.gitHandler.getCurrentlyCheckedOutBranch();
        this.gitHandler.checkoutBranch(SEARCH_BRANCH);
        this.gitHandler.pullOnCurrentBranch();
        this.gitHandler.checkoutBranch(WORK_BRANCH);
        this.gitHandler.pullOnCurrentBranch();
        this.gitHandler.checkoutBranch(currentBranch);
    }

    private void setUpRepositoryStructureForQueriesAndFetchers() throws IOException {
        StudyCatalogToFetcherConverter converter = new StudyCatalogToFetcherConverter(this.getActiveLibraryEntries(), this.preferencesService.getImportFormatPreferences(), this.preferencesService.getImporterPreferences());
        for (String query : this.getSearchQueryStrings()) {
            this.createQueryResultFolder(query);
            converter.getActiveFetchers().forEach(searchBasedFetcher -> this.createFetcherResultFile(query, (SearchBasedFetcher)searchBasedFetcher));
            this.createQueryResultFile(query);
        }
        this.createStudyResultFile();
    }

    private void createQueryResultFolder(String query) throws IOException {
        Path queryResultFolder = this.getPathToQueryDirectory(query);
        this.createFolder(queryResultFolder);
    }

    private void createFolder(Path folder) throws IOException {
        if (Files.notExists(folder, new LinkOption[0])) {
            Files.createDirectory(folder, new FileAttribute[0]);
        }
    }

    private void createFetcherResultFile(String query, SearchBasedFetcher searchBasedFetcher) {
        String fetcherName = searchBasedFetcher.getName();
        Path fetcherResultFile = this.getPathToFetcherResultFile(query, fetcherName);
        this.createBibFile(fetcherResultFile);
    }

    private void createQueryResultFile(String query) {
        Path queryResultFile = this.getPathToFetcherResultFile(query, "result");
        this.createBibFile(queryResultFile);
    }

    private void createStudyResultFile() {
        this.createBibFile(this.getPathToStudyResultFile());
    }

    private void createBibFile(Path file) {
        if (Files.notExists(file, new LinkOption[0])) {
            try {
                Files.createFile(file, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new IllegalStateException("Error during creation of repository structure.", e);
            }
        }
    }

    private String trimNameAndAddID(String query) {
        String trimmedNamed = MATCH_COLON.matcher(query).replaceAll("=");
        trimmedNamed = MATCH_ILLEGAL_CHARACTERS.matcher(trimmedNamed).replaceAll("");
        String id = this.computeIDForQuery(query);
        int remainingPathLength = 220 - this.studyDefinitionFile.toString().length() - id.length();
        if (query.length() > remainingPathLength) {
            trimmedNamed = query.substring(0, remainingPathLength);
        }
        return id + " - " + trimmedNamed;
    }

    private String computeIDForQuery(String query) {
        return String.valueOf(query.hashCode());
    }

    private void persistResults(List<QueryResult> crawlResults) throws IOException, SaveException {
        DatabaseMerger merger = new DatabaseMerger(this.preferencesService.getBibEntryPreferences().getKeywordSeparator().charValue());
        BibDatabase newStudyResultEntries = new BibDatabase();
        for (QueryResult result : crawlResults) {
            BibDatabase queryResultEntries = new BibDatabase();
            for (FetchResult fetcherResult : result.getResultsPerFetcher()) {
                BibDatabase fetcherEntries = fetcherResult.getFetchResult();
                BibDatabaseContext existingFetcherResult = this.getFetcherResultEntries(result.getQuery(), fetcherResult.getFetcherName());
                merger.merge(existingFetcherResult.getDatabase(), fetcherEntries);
                this.generateCiteKeys(existingFetcherResult, fetcherEntries);
                merger.merge(queryResultEntries, fetcherEntries);
                this.writeResultToFile(this.getPathToFetcherResultFile(result.getQuery(), fetcherResult.getFetcherName()), existingFetcherResult);
            }
            BibDatabaseContext existingQueryEntries = this.getQueryResultEntries(result.getQuery());
            merger.merge(existingQueryEntries.getDatabase(), queryResultEntries);
            merger.merge(newStudyResultEntries, queryResultEntries);
            this.writeResultToFile(this.getPathToQueryResultFile(result.getQuery()), existingQueryEntries);
        }
        BibDatabaseContext existingStudyResultEntries = this.getStudyResultEntries();
        merger.merge(existingStudyResultEntries.getDatabase(), newStudyResultEntries);
        this.writeResultToFile(this.getPathToStudyResultFile(), existingStudyResultEntries);
    }

    private void generateCiteKeys(BibDatabaseContext existingEntries, BibDatabase targetEntries) {
        CitationKeyGenerator citationKeyGenerator = new CitationKeyGenerator(existingEntries, this.preferencesService.getCitationKeyPatternPreferences());
        targetEntries.getEntries().stream().filter(bibEntry -> !bibEntry.hasCitationKey()).forEach(citationKeyGenerator::generateAndSetKey);
    }

    private void writeResultToFile(Path pathToFile, BibDatabaseContext context) throws SaveException {
        try (AtomicFileWriter fileWriter = new AtomicFileWriter(pathToFile, StandardCharsets.UTF_8);){
            SelfContainedSaveConfiguration saveConfiguration = (SelfContainedSaveConfiguration)new SelfContainedSaveConfiguration().withSaveOrder(context.getMetaData().getSaveOrder().map(so -> SelfContainedSaveOrder.of(so)).orElse(SaveOrder.getDefaultSaveOrder())).withReformatOnSave(this.preferencesService.getLibraryPreferences().shouldAlwaysReformatOnSave());
            BibWriter bibWriter = new BibWriter(fileWriter, OS.NEWLINE);
            BibtexDatabaseWriter databaseWriter = new BibtexDatabaseWriter(bibWriter, saveConfiguration, this.preferencesService.getFieldPreferences(), this.preferencesService.getCitationKeyPatternPreferences(), this.bibEntryTypesManager);
            databaseWriter.saveDatabase(context);
        }
        catch (UnsupportedCharsetException ex) {
            throw new SaveException(Localization.lang("Character encoding UTF-8 is not supported.", ex));
        }
        catch (IOException ex) {
            throw new SaveException("Problems saving", ex);
        }
    }

    private Path getPathToFetcherResultFile(String query, String fetcherName) {
        return this.repositoryPath.resolve(this.trimNameAndAddID(query)).resolve(FileNameCleaner.cleanFileName(fetcherName) + ".bib");
    }

    private Path getPathToQueryResultFile(String query) {
        return this.repositoryPath.resolve(this.trimNameAndAddID(query)).resolve("result.bib");
    }

    private Path getPathToStudyResultFile() {
        return this.repositoryPath.resolve("studyResult.bib");
    }

    private Path getPathToQueryDirectory(String query) {
        return this.repositoryPath.resolve(this.trimNameAndAddID(query));
    }
}

