/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2025 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package impl.java.io;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.StringTokenizer;
import java.util.Vector;

import blues.Container;
import blues.Log;
import blues.libbluray.LB;

public class BluesFileSystem implements Closeable {

    private static BluesFileSystem instance = null;

    HashMap<String, FsDir> roots = new HashMap();
    String[] rootNames = new String[0];
    Vector<FsFD> fds = new Vector();

    public static final int FIRST_FD = 3;

    public static synchronized BluesFileSystem getInstance() {
        if (instance == null) {
            instance = new BluesFileSystem();
            Container.my().atExit(instance);
        }
        return instance;
    }

    public synchronized void addLocalRoot(String path) {
        roots.put(path, new LocalDir(path));
        populateRootNames();
    }

    public synchronized void addRemoteRoot(String path, String nativePath) {
        roots.put(path, new BluesDir(path, nativePath));
        populateRootNames();
    }

    public synchronized void addJarRoot(String path, String nativePath) {
        roots.put(path, new JarRootDir(path, nativePath));
        populateRootNames();
    }

    public static synchronized String getDMSCCRoot() {
        return LB.getDMSCCRoot();
    }

    private void populateRootNames() {
        rootNames = new String[roots.size()];
        System.arraycopy(roots.keySet().toArray(), 0, rootNames, 0, rootNames.length);
    }

    private FsObject findObject(String path) {
        if (path == null)
            return null;
        FsObject obj = roots.get(path);
        if (obj != null)
            return obj;

        for (int i = 0; i < rootNames.length; i++) {
            String rootName = rootNames[i];
            int rootNameLen = rootName.length();

            if ((path.length() > (rootNameLen + 1)) && (path.charAt(rootNameLen) == '/')
                    && (path.startsWith(rootName))) {
                return FsDir.findEntry(roots.get(rootName), path.substring(rootNameLen + 1));
            }
        }
        return null;
    }

    private FsDir findParentDir(String path) {
        if (path == null)
            return null;
        int rindex = path.lastIndexOf('/');
        if (rindex < 1)
            return null;

        FsObject obj = findObject(path.substring(0, rindex));
        if (obj == null)
            return null;
        if (!(obj instanceof FsDir))
            return null;
        return (FsDir) obj;
    }

    private static String nameOnly(String path) {
        if (path == null)
            return null;
        int rindex = path.lastIndexOf('/');
        if (rindex < 1)
            return null;
        return path.substring(rindex + 1);
    }

    public static String canonicalize(String path) {
        if (path.length() == 0)
            return null;
        if (path.charAt(0) != '/')
            return null;

        int pathLen = path.length();
        boolean redFlag = false;
        for (int i = 0; i < pathLen; i++) {
            char c;

            c = path.charAt(i);
            if ((c == '\\') || (c == ':'))
                return null;
            if (c != '/')
                continue;

            if ((i + 1) >= pathLen) {
                redFlag = true;
                break;
            }

            c = path.charAt(i + 1);
            if ((c == '/') || (c == '.')) {
                redFlag = true;
                break;
            }
        }
        if (redFlag == false)
            return path;

        ArrayList<String> stack = new ArrayList<String>();

        StringTokenizer tk = new StringTokenizer(path, "/");
        while (tk.hasMoreTokens()) {
            String token = tk.nextToken();
            if (token.equals("."))
                continue;
            if (token.equals("..")) {
                if (stack.size() == 0)
                    return null;
                stack.remove(stack.size() - 1);
                continue;
            }
            stack.add(token);
        }

        if (stack.size() == 0)
            return "/";

        StringBuilder sb = new StringBuilder(path.length() + 1);
        for (int i = 0; i < stack.size(); i++) {
            sb.append('/');
            sb.append(stack.get(i));
        }
        String r = sb.toString();
        Log.log(Log.LOG_FILE, "canonicalize ", path, " -> ", sb.toString());
        return r;
    }

    public synchronized static int getBooleanAttributes0(String path) {
        FsObject obj = getInstance().findObject(path);
        if (null == obj) {
            Log.log(2, Log.LOG_FILE, "getBooleanAttributes0(", path, ")=<not found>");
            return 0;
        }
        int r = obj.getBooleanAttributes();
        Log.log(2, Log.LOG_FILE, "getBooleanAttributes0(", path, ")=", Integer.toString(r));
        return r;
    }

    public synchronized static boolean checkAccess(String path, int access) {
        FsObject obj = getInstance().findObject(path);
        if (null == obj) {
            Log.log(Log.LOG_FILE, "checkAccess(", path, ")=<not found>");
            return false;
        }
        boolean r = ((obj.getAccessFlags() & access) != 0);
        Log.log(Log.LOG_FILE, "checkAccess(", path, ",", Integer.toString(access), ")=", r ? "true" : "false");
        return r;
    }

    public synchronized static long getLastModifiedTime(String path) {
        FsObject obj = getInstance().findObject(path);
        if (null == obj) {
            Log.log(Log.LOG_FILE, "getLastModifiedTime(", path, ")=<not found>");
            return 0;
        }
        long r = obj.getLastModifiedTime();
        Log.log(Log.LOG_FILE, "getLastModifiedTime(", path, ")=", Long.toString(r));
        return r;
    }

    public synchronized static long getLength(String path) {
        FsObject obj = getInstance().findObject(path);
        if (null == obj) {
            Log.log(2, Log.LOG_FILE, "getLength(", path, ")=<not found>");
            return 0;
        }
        if (!(obj instanceof FsFile)) {
            Log.log(2, Log.LOG_FILE, "getLength(", path, ")=<directory>");
            return 0;
        }
        long r = ((FsFile) obj).getLength();
        Log.log(2, Log.LOG_FILE, "getLength(", path, ")=", Long.toString(r));
        return r;
    }

    public synchronized static int createFileExclusively(String path) {
        FsDir dir = getInstance().findParentDir(path);
        if (null == dir) {
            Log.log(Log.LOG_FILE, "createFileExclusively(", path, ")=<parent dir not found>");
            return -1;
        }
        String name = nameOnly(path);
        FsObject obj = dir.findEntry(name);
        if (obj != null) {
            Log.log(Log.LOG_FILE, "createFileExclusively(", path, ")=<exists>");
            return 1;
        }

        boolean r = dir.createFileExclusively(name);
        Log.log(Log.LOG_FILE, "createFileExclusively(", path, ")=", Boolean.toString(r));
        return r ? 0 : -1;
    }

    public synchronized static boolean delete0(String path) {
        FsDir dir = getInstance().findParentDir(path);
        if (null == dir) {
            Log.log(Log.LOG_FILE, "delete0(", path, ")=<parent dir not found>");
            return false;
        }
        String name = nameOnly(path);
        FsObject obj = dir.findEntry(name);
        if (obj == null) {
            Log.log(Log.LOG_FILE, "delete0(", path, ")=<not found>");
            return false;
        }
        boolean r = dir.delete(name);
        Log.log(Log.LOG_FILE, "delete0(", path, ")=", Boolean.toString(r));
        return r;
    }

    public synchronized static String[] list(String path) {
        FsObject obj = getInstance().findObject(path);
        if (null == obj) {
            Log.log(Log.LOG_FILE, "list(", path, ")=<not found>");
            return null;
        }
        if (!(obj instanceof FsDir)) {
            Log.log(Log.LOG_FILE, "list(", path, ")=<not a dir>");
            return null;
        }

        String[] names = ((FsDir) obj).list();
        Log.log(Log.LOG_FILE, names, "list(", path, ")=");
        return names;
    }

    public synchronized static boolean createDirectory(String path) {
        FsDir dir = getInstance().findParentDir(path);
        if (null == dir) {
            Log.log(Log.LOG_FILE, "createDirectory(", path, ")=<parent dir not found>");
            return false;
        }
        String name = nameOnly(path);
        FsObject obj = dir.findEntry(name);
        if (obj != null) {
            Log.log(Log.LOG_FILE, "createDirectory(", path, ")=<exists>");
            return false;
        }

        boolean r = dir.createDirectory(name);
        Log.log(Log.LOG_FILE, "createDirectory(", path, ")=", Boolean.toString(r));
        return r;
    }

    public synchronized static boolean rename0(String path1, String path2) {
        Log.log(Log.LOG_FILE, "rename0(", path1, ",", path2, ")=<unimplemented>");
        return false;
    }

    public synchronized static boolean setLastModifiedTime(String path, long time) {
        FsObject obj = getInstance().findObject(path);
        if (null == obj) {
            Log.log(Log.LOG_FILE, "setLastModifiedTime(", path, ")=<not found>");
            return false;
        }
        if ((obj.getAccessFlags() & FsObject.ACCESS_WRITE) != 0) {
            obj.setLastModifiedTime(time);
            Log.log(Log.LOG_FILE, "setLastModifiedTime(", path, ")=", Long.toHexString(time));
            return true;
        }
        Log.log(Log.LOG_FILE, "setLastModifiedTime(", path, ")=<read-only>");
        return false;
    }

    public synchronized static boolean setReadOnly(String path) {
        FsObject obj = getInstance().findObject(path);
        if (null == obj) {
            Log.log(Log.LOG_FILE, "setReadOnly(", path, ")=<not found>");
            return false;
        }
        if ((obj.getAccessFlags() & FsObject.ACCESS_WRITE) != 0) {
            obj.setAccessFlags(obj.getAccessFlags() ^ FsObject.ACCESS_WRITE);
            Log.log(Log.LOG_FILE, "setReadOnly(", path, ")=true");
            return true;
        }
        Log.log(Log.LOG_FILE, "setReadOnly(", path, ")=<already read-only>");
        return false;
    }

    public synchronized static int open(String path, boolean write, boolean create, boolean append) {
        FsObject obj = getInstance().findObject(path);
        if (obj == null) {
            if (create == false) {
                Log.log(2, Log.LOG_FILE, "open(", path, ")=<not found>");
                return -1;
            }
            FsDir dir = getInstance().findParentDir(path);
            if (dir == null) {
                Log.log(2, Log.LOG_FILE, "open(", path, ")=<parent dir not found>");
                return -1;
            }
            String name = nameOnly(path);
            if (dir.createFileExclusively(name) == false) {
                Log.log(2, Log.LOG_FILE, "open(", path, ")=<unable to create file>");
                return -1;
            }
            obj = dir.findEntry(name);
        }
        if (!(obj instanceof FsFile)) {
            Log.log(2, Log.LOG_FILE, "open(", path, ")=<not a file>");
            return -1;
        }
        if (write) {
            if ((obj.getAccessFlags() & FsObject.ACCESS_WRITE) == 0) {
                Log.log(2, Log.LOG_FILE, "open(", path, ")=<attempt to write read-only file>");
                return -1;
            }
        }

        int r = 0;
        FsFD fd = new FsFD((FsFile) obj, write, append);
        for (int i = 0; i < getInstance().fds.size(); i++) {
            if (getInstance().fds.get(i) == null) {
                getInstance().fds.set(i, fd);
                r = i + FIRST_FD;
                break;
            }
        }
        if (r == 0) {
            r = getInstance().fds.size() + FIRST_FD;
            getInstance().fds.addElement(fd);
        }
        Log.log(2, Log.LOG_FILE, "open(", path, ")=", Integer.toString(r));
        return r;
    }

    private FsFD findFD(int fd) {
        if (fd < FIRST_FD)
            return null;
        int index = fd - FIRST_FD;
        if (index >= fds.size())
            return null;
        return fds.get(index);
    }

    public synchronized static int readBytes(int fd, byte b[], int off, int len) {
        FsFD fdobj = getInstance().findFD(fd);
        if (fdobj == null)
            return -2;
        int r = fdobj.readBytes(b, off, len);
        Log.log(2, Log.LOG_FILE, "readBytes(", Integer.toString(fd), ",", Integer.toString(len), ")=",
                Integer.toString(r));
        return r;
    }

    public synchronized static int writeBytes(int fd, byte b[], int off, int len) {
        FsFD fdobj = getInstance().findFD(fd);
        if (fdobj == null)
            return -2;
        int r = fdobj.writeBytes(b, off, len);
        Log.log(2, Log.LOG_FILE, "writeBytes(", Integer.toString(fd), ",", Integer.toString(len), ")=",
                Integer.toString(r));
        return r;
    }

    public synchronized static long seek(int fd, long pos, boolean relative) {
        FsFD fdobj = getInstance().findFD(fd);
        if (fdobj == null)
            return -2;
        long r = fdobj.seek(pos, relative);
        Log.log(Log.LOG_FILE, "seek(", Integer.toString(fd), ",", Long.toString(pos), ")=", Long.toString(r));
        return r;
    }

    public synchronized static int setLength(int fd, long newLength) {
        FsFD fdobj = getInstance().findFD(fd);
        if (fdobj == null)
            return -2;

        int r = fdobj.setLength(newLength);
        Log.log(Log.LOG_FILE, "setLength(", Integer.toString(fd), ",", Long.toString(newLength), ")=",
                Integer.toString(r));
        return r;
    }

    public synchronized static long getLength(int fd) {
        FsFD fdobj = getInstance().findFD(fd);
        if (fdobj == null)
            return -2;

        long r = fdobj.getLength();
        Log.log(Log.LOG_FILE, "getLength(", Integer.toString(fd), ")=", Long.toString(r));
        return r;
    }

    public synchronized static int available(int fd) {
        FsFD fdobj = getInstance().findFD(fd);
        if (fdobj == null)
            return -2;

        int r = fdobj.available();
        Log.log(Log.LOG_FILE, "available(", Integer.toString(fd), ")=", Integer.toString(r));
        return r;
    }

    public synchronized static int close(int fd) {
        FsFD fdobj = getInstance().findFD(fd);
        if (fdobj == null) {
            Log.log(2, Log.LOG_FILE, "close(", Integer.toString(fd), ")=<invalid fd>");
            return -2;
        }
        fdobj.close();
        int index = fd - FIRST_FD;
        getInstance().fds.set(index, null);
        Log.log(2, Log.LOG_FILE, "close(", Integer.toString(fd), ")=ok");
        return 0;
    }

    public synchronized static String getLocalPath(String jailPath) {
        FsObject fsobj = getInstance().findObject(canonicalize(jailPath));
        if (null == fsobj)
            return null;
        return getLocalPath(fsobj,jailPath);
    }

    public synchronized static String getLocalPath(FsObject fsobj,String jailPath) {

        if (fsobj instanceof LocalFile) {
            String r = ((LocalFile) fsobj).getLocalPath();
            Log.log(2, Log.LOG_FILE, "getLocalPath(", jailPath, ")=(local) ",r);
            return r;
        }

        if (fsobj instanceof BluesFile) {
            String r = cacheFile((FsFile) fsobj);
            Log.log(2, Log.LOG_FILE, "getLocalPath(", jailPath, ")=(cached) ",r);
            return r;
        }

        Log.unimplemented();
        return null;
    }

    public synchronized static String getLocalPath(InputStream fis,String name) {
        String r = cacheFile(fis,name);
        Log.log(2, Log.LOG_FILE, "getLocalPath(", fis.toString(),",",name,")=(cached) ",r);
        return r;
    }

    private static HashMap<FsFile,String> cacheMap = new HashMap<FsFile,String>();

    private static synchronized String cacheFile(FsFile src) {

        String localPath = cacheMap.get(src);
        if (localPath!=null) {
            return localPath;
        }

        java.io.File tempFile = Container.getTempName(src.getName());
        java.io.FileOutputStream fos = null;
        try {
            fos = new java.io.FileOutputStream(tempFile);
            long offset = 0;
            int BufferLen = 1 * 1024 * 1024;
            if (BufferLen > src.getLength()) {
                BufferLen = (int) src.getLength();
            }
            byte[] buffer = new byte[BufferLen];
            while (offset < src.getLength()) {
                int len = buffer.length;
                if (len > (src.length - offset)) {
                    len = (int) (src.length - offset);
                }
                int r = src.readBytes(offset, buffer, 0, len);
                if (r <= 0)
                    throw new IOException("Read error");
                fos.write(buffer, 0, r);
                offset += r;
            }
            fos.flush();
            fos.close();
            fos = null;
        } catch (IOException e) {
            cleanupFile(tempFile, fos);
            throw new RuntimeException("Unable to cache file " + src.getName(), e);
        } finally {
            if (fos!=null) {
                try {
                    fos.close();
                } catch (IOException e) {
                    cleanupFile(tempFile, fos);
                    throw new RuntimeException("Unable to cache file " + src.getName(), e);
                }
            }
        }
        localPath = tempFile.getPath();
        cacheMap.put(src, localPath);
        return localPath;
    }

    private static String cacheFile(InputStream src,String name) {
        java.io.File tempFile = Container.getTempName(name);
        java.io.FileOutputStream fos = null;
        try {
            fos = new java.io.FileOutputStream(tempFile);
            int BufferLen = 1 * 1024 * 1024;
            byte[] buffer = new byte[BufferLen];
            while (true) {
                int r = src.read(buffer, 0, BufferLen);
                if (r <= 0) {
                    break;
                }
                fos.write(buffer, 0, r);
            }
            fos.flush();
            fos.close();
            fos = null;
        } catch (IOException e) {
            cleanupFile(tempFile, fos);
            throw new RuntimeException("Unable to cache file " + name, e);
        } finally {
            if (fos!=null) {
                try {
                    fos.close();
                } catch (IOException e) {
                    cleanupFile(tempFile, fos);
                    throw new RuntimeException("Unable to cache file " + name, e);
                }
            }
        }
        return tempFile.getPath();
    }

    private static void cleanupFile(java.io.File file, java.io.FileOutputStream fos) {
        try {
            if (fos != null)
                fos.close();
        } catch (IOException e) {
        }
        if (file != null)
            file.delete();
    }

    public synchronized void close() {
        for (String rootName : rootNames) {
            FsDir rootDir = roots.get(rootName);
            rootDir.close();
        }
        roots.clear();
        rootNames = null;
    }

    public static byte[] readJailFile(String path,int maxFileSize) throws IOException {
        jail.java.io.File file = new jail.java.io.File(path);

        long size = file.length();

        if (size > maxFileSize) {
            throw new IOException("file too big");
        }

        byte[] data = new byte[(int)size];

        jail.java.io.FileInputStream fis=null;

        try {
            fis = new jail.java.io.FileInputStream(file);
            int offset=0,rest=data.length,len;
            while(rest!=0) {
                len = fis.read(data, offset, rest);
                offset += len;
                rest -= len;
            }
            fis.close();
        } finally {
            if (fis!=null) {
                fis.close();
            }
        }
        return data;
    }

    public static byte[] readLocalFile(String path,int maxFileSize) throws IOException {
        java.io.File file = new java.io.File(path);

        long size = file.length();

        if (size > maxFileSize) {
            throw new IOException("file too big");
        }

        byte[] data = new byte[(int)size];

        java.io.FileInputStream fis=null;

        try {
            fis = new java.io.FileInputStream(file);
            int offset=0,rest=data.length,len;
            while(rest!=0) {
                len = fis.read(data, offset, rest);
                offset += len;
                rest -= len;
            }
            fis.close();
        } finally {
            if (fis!=null) {
                fis.close();
            }
        }
        return data;
    }

}
