/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.gui.theme;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Path;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.function.Consumer;
import javafx.application.ColorScheme;
import javafx.application.Platform;
import javafx.collections.ObservableList;
import javafx.scene.Scene;
import javafx.scene.web.WebEngine;
import org.jabref.gui.theme.StyleSheet;
import org.jabref.gui.theme.Theme;
import org.jabref.gui.util.BindingsHelper;
import org.jabref.gui.util.UiTaskExecutor;
import org.jabref.model.util.FileUpdateListener;
import org.jabref.model.util.FileUpdateMonitor;
import org.jabref.preferences.WorkspacePreferences;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ThemeManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(ThemeManager.class);
    private final WorkspacePreferences workspacePreferences;
    private final FileUpdateMonitor fileUpdateMonitor;
    private final Consumer<Runnable> updateRunner;
    private final StyleSheet baseStyleSheet;
    private Theme theme;
    private Scene mainWindowScene;
    private final Set<WebEngine> webEngines = Collections.newSetFromMap(new WeakHashMap());

    public ThemeManager(WorkspacePreferences workspacePreferences, FileUpdateMonitor fileUpdateMonitor, Consumer<Runnable> updateRunner) {
        this.workspacePreferences = Objects.requireNonNull(workspacePreferences);
        this.fileUpdateMonitor = Objects.requireNonNull(fileUpdateMonitor);
        this.updateRunner = Objects.requireNonNull(updateRunner);
        this.baseStyleSheet = StyleSheet.create("Base.css").get();
        this.theme = workspacePreferences.getTheme();
        this.addStylesheetToWatchlist(this.baseStyleSheet, this::baseCssLiveUpdate);
        this.baseCssLiveUpdate();
        BindingsHelper.subscribeFuture(workspacePreferences.themeProperty(), theme -> this.updateThemeSettings());
        BindingsHelper.subscribeFuture(workspacePreferences.themeSyncOsProperty(), theme -> this.updateThemeSettings());
        BindingsHelper.subscribeFuture(workspacePreferences.shouldOverrideDefaultFontSizeProperty(), should -> this.updateFontSettings());
        BindingsHelper.subscribeFuture(workspacePreferences.mainFontSizeProperty(), size -> this.updateFontSettings());
        BindingsHelper.subscribeFuture(Platform.getPreferences().colorSchemeProperty(), colorScheme -> this.updateThemeSettings());
        this.updateThemeSettings();
    }

    private void updateThemeSettings() {
        Theme newTheme = Objects.requireNonNull(this.workspacePreferences.getTheme());
        if (this.workspacePreferences.themeSyncOsProperty().getValue().booleanValue()) {
            newTheme = Platform.getPreferences().getColorScheme() == ColorScheme.DARK ? Theme.dark() : Theme.light();
        }
        if (newTheme.equals(this.theme)) {
            LOGGER.info("Not updating theme because it hasn't changed");
        } else {
            this.theme.getAdditionalStylesheet().ifPresent(this::removeStylesheetFromWatchList);
        }
        this.theme = newTheme;
        LOGGER.info("Theme set to {} with base css {}", (Object)newTheme, (Object)this.baseStyleSheet);
        this.theme.getAdditionalStylesheet().ifPresent(styleSheet -> this.addStylesheetToWatchlist((StyleSheet)styleSheet, this::additionalCssLiveUpdate));
        this.additionalCssLiveUpdate();
        this.updateFontSettings();
    }

    private void updateFontSettings() {
        UiTaskExecutor.runInJavaFXThread(() -> this.updateRunner.accept(() -> this.updateFontStyle(this.mainWindowScene)));
    }

    private void removeStylesheetFromWatchList(StyleSheet styleSheet) {
        Path oldPath = styleSheet.getWatchPath();
        if (oldPath != null) {
            this.fileUpdateMonitor.removeListener(oldPath, this::additionalCssLiveUpdate);
            LOGGER.info("No longer watch css {} for live updates", (Object)oldPath);
        }
    }

    private void addStylesheetToWatchlist(StyleSheet styleSheet, FileUpdateListener updateMethod) {
        Path watchPath = styleSheet.getWatchPath();
        if (watchPath != null) {
            try {
                this.fileUpdateMonitor.addListenerForFile(watchPath, updateMethod);
                LOGGER.info("Watching css {} for live updates", (Object)watchPath);
            }
            catch (IOException e) {
                LOGGER.warn("Cannot watch css path {} for live updates", (Object)watchPath, (Object)e);
            }
        }
    }

    private void baseCssLiveUpdate() {
        this.baseStyleSheet.reload();
        if (this.baseStyleSheet.getSceneStylesheet() == null) {
            LOGGER.error("Base stylesheet does not exist.");
        } else {
            LOGGER.debug("Updating base CSS for main window scene");
        }
        UiTaskExecutor.runInJavaFXThread(() -> this.updateRunner.accept(this::updateBaseCss));
    }

    private void additionalCssLiveUpdate() {
        String newStyleSheetLocation = this.theme.getAdditionalStylesheet().map(styleSheet -> {
            styleSheet.reload();
            return styleSheet.getWebEngineStylesheet();
        }).orElse("");
        LOGGER.debug("Updating additional CSS for main window scene and {} web engines", (Object)this.webEngines.size());
        UiTaskExecutor.runInJavaFXThread(() -> this.updateRunner.accept(() -> {
            this.updateAdditionalCss();
            this.webEngines.forEach(webEngine -> {
                if (newStyleSheetLocation.equals(webEngine.getUserStyleSheetLocation())) {
                    webEngine.setUserStyleSheetLocation(null);
                }
                webEngine.setUserStyleSheetLocation(newStyleSheetLocation);
            });
        }));
    }

    private void updateBaseCss() {
        if (this.mainWindowScene == null) {
            return;
        }
        ObservableList stylesheets = this.mainWindowScene.getStylesheets();
        if (!stylesheets.isEmpty()) {
            stylesheets.removeFirst();
        }
        stylesheets.addFirst(this.baseStyleSheet.getSceneStylesheet().toExternalForm());
    }

    private void updateAdditionalCss() {
        if (this.mainWindowScene == null) {
            return;
        }
        this.mainWindowScene.getStylesheets().setAll(List.of(this.baseStyleSheet.getSceneStylesheet().toExternalForm(), this.theme.getAdditionalStylesheet().map(styleSheet -> {
            URL stylesheetUrl = styleSheet.getSceneStylesheet();
            if (stylesheetUrl != null) {
                return stylesheetUrl.toExternalForm();
            }
            return "";
        }).orElse("")));
    }

    public void installCss(Scene mainWindowScene) {
        Objects.requireNonNull(mainWindowScene, "scene is required");
        this.updateRunner.accept(() -> {
            this.mainWindowScene = mainWindowScene;
            this.updateBaseCss();
            this.updateAdditionalCss();
        });
    }

    public void installCss(WebEngine webEngine) {
        this.updateRunner.accept(() -> {
            if (this.webEngines.add(webEngine)) {
                webEngine.setUserStyleSheetLocation(this.theme.getAdditionalStylesheet().isPresent() ? this.theme.getAdditionalStylesheet().get().getWebEngineStylesheet() : "");
            }
        });
    }

    public void updateFontStyle(Scene scene) {
        if (scene == null) {
            return;
        }
        if (this.workspacePreferences.shouldOverrideDefaultFontSize()) {
            scene.getRoot().setStyle("-fx-font-size: " + this.workspacePreferences.getMainFontSize() + "pt;");
        } else {
            scene.getRoot().setStyle("-fx-font-size: " + this.workspacePreferences.getDefaultFontSize() + "pt;");
        }
    }

    @VisibleForTesting
    Theme getActiveTheme() {
        return this.theme;
    }
}

